import 'source-map-support/register';
import { DefineSchema, RegisterSchema } from 'koishi-thirdeye';
import _ from 'lodash';
import { Adapter, Bot, Logger, Random } from 'koishi';
import moment from 'moment';
import { createHash, Hash } from 'crypto';

const logger = new Logger('greeting');

interface Salt {
  salt?: string;
}

function updateSalt(hash: Hash, saltObjects: Salt[]) {
  for (const saltObject of saltObjects) {
    if (saltObject?.salt) {
      hash.update(saltObject.salt);
    }
  }
  return hash;
}

function hashRandom(saltObjects: Salt[]) {
  const hash = updateSalt(createHash('md5'), saltObjects);
  return parseInt(hash.digest('hex'), 16) % 4294967295;
}

@RegisterSchema()
export class Word {
  constructor(config: Partial<Word>) {}

  @DefineSchema({ description: '文本。', required: true })
  word: string;

  @DefineSchema({ description: '权重。', default: 1 })
  weight?: number;

  toEntry() {
    return [this.word, this.weight || 1];
  }
}

@RegisterSchema()
export class Pattern {
  constructor(config: Partial<Pattern>) {}

  @DefineSchema({
    description: '词语。该语段中会随机悬着其中1种词语出现。',
    type: Word,
  })
  words: Word[];

  pickWord() {
    return Random.weightedPick(
      Object.fromEntries(this.words.map((word) => word.toEntry())),
    );
  }
}

@RegisterSchema()
export class Rule implements Salt {
  constructor(config: Partial<Rule>) {}

  @DefineSchema({ description: '最早出现的时间。', default: '00:00' })
  fromTime?: string;

  @DefineSchema({ description: '最晚出现的时间。默认为与 fromTime 相同。' })
  toTime?: string;

  getTimeRange() {
    const nowDate = moment().format('YYYY-MM-DD');
    const fromTime = moment(`${nowDate} ${this.fromTime}`);
    const toTime = moment(`${nowDate} ${this.toTime || this.fromTime}`);
    return [fromTime, toTime] as const;
  }

  shouldSend(target: Target) {
    const now = moment();
    const [fromTime, toTime] = this.getTimeRange();
    const fromMinutes = Math.floor(fromTime.unix() / 60);
    const toMinutes = Math.floor(toTime.unix() / 60);
    const nowMinutes = Math.floor(now.unix() / 60);
    let sendingMinutes: number;
    if (fromMinutes === toMinutes) {
      sendingMinutes = fromMinutes;
    } else {
      const scale = toMinutes - fromMinutes + 1;
      const random =
        hashRandom([this, target, { salt: now.format('YYYY-MM-DD') }]) % scale;
      sendingMinutes = fromMinutes + random;
    }
    const sendingTime = moment.unix(sendingMinutes * 60);
    logger.debug(
      `${nowMinutes}: Should send ${this.generateText()} to ${target.getDescription()} at ${sendingTime.format(
        'YYYY-MM-DD HH:mm',
      )}`,
    );
    return nowMinutes === sendingMinutes;
  }

  @DefineSchema({ description: '随机时间盐值。' })
  salt?: string;

  @DefineSchema({ description: '语段。', type: Pattern })
  patterns: Pattern[];

  generateText() {
    return this.patterns.map((pattern) => pattern.pickWord()).join('');
  }
}

@RegisterSchema()
export class Target implements Salt {
  constructor(config: Partial<Target>) {}

  @DefineSchema({ description: '私聊用户 ID。若存在此项则为私聊。' })
  userId?: string;

  @DefineSchema({ description: '频道 ID。' })
  channelId?: string;

  @DefineSchema({ description: '群组 ID。' })
  guildId?: string;

  getDescription() {
    if (this.userId) {
      return `User ${this.userId}`;
    } else if (this.channelId) {
      return `Channel ${this.channelId}`;
    } else if (this.guildId) {
      return `Group ${this.guildId}`;
    } else {
      return 'Unknown target';
    }
  }

  get salt() {
    return this.userId || this.channelId || this.guildId;
  }

  set salt(s: string) {}

  send(bot: Bot, text: string) {
    if (this.userId) {
      return bot.sendPrivateMessage(this.userId, text);
    } else {
      return bot.sendMessage(this.channelId, text, this.guildId);
    }
  }
}

@RegisterSchema()
export class Instance {
  constructor(config: Partial<Instance>) {}

  @DefineSchema({
    description:
      '机器人 ID，应写成 platform:id 的形式。例如 onebot:111111111。',
    required: true,
  })
  from: string;

  getBot(list: Adapter.BotList) {
    return list.get(this.from);
  }

  @DefineSchema({
    description: '目标列表。',
    type: Target,
  })
  to: Target[];

  @DefineSchema({
    description: '规则列表。',
    type: Rule,
  })
  rules: Rule[];

  async run(list: Adapter.BotList) {
    const bot = this.getBot(list);
    if (bot?.status !== 'online') {
      logger.info(`Skipping job for ${this.from} because it's not online.`);
      return this;
    }
    logger.debug(
      `${moment().format('YYYY-MM-DD HH:mm')}: Running job for ${this.from}...`,
    );
    const prom: Promise<string[]>[] = [];
    for (const target of this.to) {
      for (const rule of this.rules) {
        if (rule.shouldSend(target)) {
          const text = rule.generateText();
          logger.info(
            `Sending from ${this.from} to ${target.getDescription()}: ${text}`,
          );
          prom.push(
            target.send(bot, text).catch((e) => {
              logger.error(
                `Failed to send from ${
                  this.from
                } to ${target.getDescription()}: ${e.message}`,
              );
              return undefined;
            }),
          );
        }
      }
    }
    return;
  }
}

@RegisterSchema()
export class GreetingPluginConfig {
  constructor(config: GreetingPluginConfigLike) {}

  @DefineSchema({ description: '实例列表。', type: Instance })
  instances: Instance[];

  async runInstances(list: Adapter.BotList, instances = this.instances) {
    const failedInstances = _.compact(
      await Promise.all(instances.map((instance) => instance.run(list))),
    );
    return failedInstances;
  }
}

export type GreetingPluginConfigLike = Partial<GreetingPluginConfig>;
