import { EdgingPluginConfig, EdgingProfile } from './config';
import {
  DefinePlugin,
  StarterPlugin,
  Provide,
  Inject,
  LifecycleEvents,
  UseCommand,
  InjectLogger,
  PutObject,
  PutUserId,
  PutOption,
  UseMiddleware,
  OnPrivate,
  PutBot,
  selectContext,
} from 'koishi-thirdeye';
import { Adapter, Bot, Logger, Next, Session } from 'koishi';
import type PicsContainer from 'koishi-plugin-pics';
import { dynamicInterval } from './utility';
export * from './config';

declare module 'koishi' {
  interface Context {
    edging: EdgingPlugin;
  }
}

class UserDef {
  @PutUserId()
  operatorId: string;

  @PutOption('userId', '-u <userId>  目标用户 ID。')
  userId: string;

  getUserId() {
    return this.userId || this.operatorId;
  }
}

@DefinePlugin()
export class EdgingPanel {
  @Inject('edging')
  main: Partial<EdgingPlugin>;

  @UseCommand('edging', '寸止的配置。', { empty: true })
  edgingCommand() {}

  @UseCommand('edging.start', 'Start edging.')
  start(@PutObject() profile: EdgingProfile) {
    return this.main.start(profile);
  }

  @UseCommand('edging.stop', 'Stop edging.')
  stop(@PutObject() user: UserDef) {
    return this.main.stop(user.getUserId());
  }

  @UseCommand('edging.finish', 'Finish edging.')
  finish(@PutObject() user: UserDef) {
    return this.main.finish(user.getUserId());
  }

  @UseCommand('edging.status', 'Show edging status.')
  status(@PutBot() bot: Bot) {
    return this.main.status(bot);
  }
}

@Provide('edging', { immediate: true })
@DefinePlugin({ name: 'edging', schema: EdgingPluginConfig })
export default class EdgingPlugin
  extends StarterPlugin(EdgingPluginConfig)
  implements LifecycleEvents
{
  onApply() {
    selectContext(this.ctx, this.config.panelSelection).plugin(EdgingPanel);
    dynamicInterval(
      this.ctx,
      this.wrapTry(() => this.processText(), 'process text'),
      this.config.minTextInterval,
      this.config.maxTextInterval,
    );
    dynamicInterval(
      this.ctx,
      this.wrapTry(() => this.processPics(), 'process pic'),
      this.config.minPicInterval,
      this.config.maxPicInterval,
    );
  }

  @Inject()
  private bots: Bot[];

  @Inject()
  private pics: PicsContainer;

  @InjectLogger()
  private logger: Logger;

  profiles = new Map<string, EdgingProfile>();

  endPlay(profile: EdgingProfile) {
    this.logger.info(
      `Play from ${profile.bot.sid} to ${profile.getTarget()} finished.`,
    );
    this.profiles.delete(profile.getTarget());
  }

  async checkOvertime(profile: EdgingProfile) {
    if (profile.isOvertime()) {
      this.endPlay(profile);
      await profile.success();
      return true;
    }
    return false;
  }

  async runForEachProfile<T>(
    fn: (profile: EdgingProfile) => T | Promise<T>,
    action = 'run',
    profiles = Array.from(this.profiles.values()),
  ) {
    return Promise.all(
      profiles.map((profile) => {
        const wrapper = this.wrapTry(
          fn,
          `${action} for profile ${profile.getTarget()}`,
        );
        return wrapper(profile);
      }),
    );
  }

  async processText() {
    this.logger.info(`Processing text...`);
    return this.runForEachProfile(async (profile) => {
      this.logger.info(`Processing text for profile ${profile.getTarget()}...`);
      if (await this.checkOvertime(profile)) {
        return;
      }
      return profile.process();
    }, 'process text');
  }

  async processPics() {
    if (!this.pics) {
      return;
    }
    const profiles = Array.from(this.profiles.values()).filter((p) =>
      p.getUsePics(),
    );
    if (!profiles.length) {
      return;
    }
    this.logger.info(`Processing pics...`);
    const pic = await this.pics.randomPic(
      this.config.picTags,
      this.config.sourceTags,
    );
    this.logger.info(`Get pic ${pic.url}`);
    if (!pic) {
      this.logger.warn(`No pic found.`);
      return;
    }
    const segment = await this.pics.getSegment(pic.url);
    return this.runForEachProfile(
      async (profile) => {
        this.logger.info(
          `Processing pic for profile ${profile.getTarget()}...`,
        );
        if (await this.checkOvertime(profile)) {
          return;
        }
        return profile.send(segment.toString());
      },
      'process pic',
      profiles,
    );
  }

  @OnPrivate()
  @UseMiddleware()
  async checkFinish(session: Session, next: Next) {
    const profile = this.profiles.get(session.userId);
    if (!profile || session.bot !== profile.bot) {
      return next();
    }
    if (profile.isFail(session.content)) {
      await profile.fail();
      this.endPlay(profile);
      return;
    }
    return next();
  }

  private wrapTry<Args extends any[], T>(
    fn: (...args: Args) => T | Promise<T>,
    action = 'perform action',
  ) {
    return async (...args: Args) => {
      try {
        return await fn(...args);
      } catch (e) {
        this.logger.error(`Failed to ${action}: ${e.message}`);
      }
    };
  }

  async start(profile: EdgingProfile) {
    if (this.profiles.has(profile.getTarget())) {
      return `${profile.getTarget()} 已经在玩耍了。`;
    }
    const errorMessage = await profile.initialize(this.config, this.bots);
    if (errorMessage) {
      return errorMessage;
    }
    this.profiles.set(profile.getTarget(), profile);
    this.logger.info(
      `Started play from ${profile.bot.sid} to ${profile.getTarget()}`,
    );
    return '准备好玩耍了呢。';
  }

  async stop(userId: string) {
    const profile = this.profiles.get(userId);
    if (!profile) {
      return `${userId} 没有在玩耍。`;
    }
    this.endPlay(profile);
    return '停止玩耍了呢。';
  }

  async finish(userId: string) {
    const profile = this.profiles.get(userId);
    if (!profile) {
      return `${userId} 没有在玩耍。`;
    }
    this.endPlay(profile);
    await profile.fail();
    return '玩坏了呢。';
  }

  status(bot: Bot) {
    const profiles = Array.from(this.profiles.values()).filter(
      (profile) => profile.bot === bot,
    );
    return `${profiles.map((profile) => profile.describe())}我一共在和 ${
      profiles.length
    } 个人玩耍。`;
  }
}
