// import 'source-map-support/register';
import {
  SchemaProperty,
  RegisterSchema,
  PutOption,
  SchemaOptions,
  PutSession,
  Selection,
} from 'koishi-thirdeye';
import { Adapter, Bot, Random, Session } from 'koishi';
import path from 'path';
import * as fs from 'fs';
import moment from 'moment';
import { sendPriv } from './utility';

const ProfileProperty = (
  desc: string,
  schema?: SchemaOptions,
): PropertyDecorator => {
  return (target, key: string) => {
    SchemaProperty({ description: desc.split('  ')[1], ...schema })(
      target,
      key,
    );
    PutOption(key, desc)(target, key);
  };
};

export class EdgingProfile {
  @PutSession()
  session: Session;

  @ProfileProperty('-b <from>  来源。')
  from: string;

  bot: Bot;

  getBot(bots: Adapter.BotList) {
    if (this.from) {
      return bots.get(this.from);
    } else {
      return this.session.bot;
    }
  }

  @ProfileProperty('-u <to>  目标。')
  to: string;

  @ProfileProperty('-d <duration>  时间，单位秒。', { default: 600 })
  duration: number;

  isOvertime() {
    return moment().diff(this.startTime, 'seconds') > this.duration;
  }

  getTarget() {
    if (this.to) {
      return this.to;
    } else {
      return this.session.userId;
    }
  }

  @ProfileProperty('-m <model>  使用模板。')
  model: string;

  @ProfileProperty('--encoding <encoding>  文件编码。', {
    type: String,
    default: 'utf8',
  })
  encoding: BufferEncoding;

  async loadModel(root: string) {
    const filename = path.join(root, `${this.model}.txt`);
    const content = await fs.promises.readFile(filename, {
      encoding: this.encoding,
    });
    this.texts = content
      .split(/\r?\n/)
      .map((line) => line.trim())
      .filter((line) => line.length > 0);
  }

  texts: string[];

  @SchemaProperty({ type: String, default: [''], description: '随机前缀。' })
  prefixes: string[];

  @SchemaProperty({ type: String, default: [''], description: '随机后缀。' })
  suffixes: string[];

  startTime: moment.Moment;

  @ProfileProperty('-p  使用图片。')
  usePics: boolean;

  @ProfileProperty('-n  不使用图片。')
  noUsePics: boolean;

  getUsePics() {
    return this.usePics && !this.noUsePics;
  }

  previousText = '';

  attachText(text: string) {
    return `${Random.pick(this.prefixes)}${text}${Random.pick(this.suffixes)}`;
  }

  getText() {
    const texts =
      this.texts.length > 1
        ? this.texts.filter((text) => text !== this.previousText)
        : this.texts;
    const text = Random.pick(texts);
    this.previousText = text;
    return this.attachText(text);
  }

  @ProfileProperty('--fail-text <tags>  败北时的消息。')
  failText: string;

  @ProfileProperty('--success-text <tags>  成功时的消息。')
  successText: string;

  @ProfileProperty('--fail-trigger-word <word>  败北检测词。')
  failTriggerWord: string;

  isFail(message: string) {
    return this.failTriggerWord && message.startsWith(this.failTriggerWord);
  }

  async initialize(config: EdgingPluginConfig, bots: Adapter.BotList) {
    Object.keys(config.defaultProfile).forEach((key) => {
      if (!this[key] && config.defaultProfile[key]) {
        this[key] = config.defaultProfile[key];
      }
    });
    if (!this.model) {
      return '没有指定模板呢。';
    }
    this.bot = this.getBot(bots);
    if (!this.bot) {
      return `没有找到陪着玩的 ${this.from} 呢。`;
    }
    try {
      await this.loadModel(config.modelRoot);
    } catch (e) {
      return `模板似乎难产了: ${e.message}`;
    }
    this.startTime = moment();
  }

  send(content: string) {
    const bot = this.bot;
    if (
      bot &&
      (bot.platform === 'onebot' ||
        (bot.platform === 'qqguild' && bot['parentBot']?.platform === 'onebot'))
    ) {
      content = content.replace(/,url=base64/g, ',file=base64');
    }
    return sendPriv(bot, this.getTarget(), content);
  }

  process() {
    return this.send(this.getText());
  }

  fail() {
    if (!this.failText) {
      return;
    }
    return this.send(this.attachText(this.failText));
  }

  success() {
    if (!this.successText) {
      return;
    }
    return this.send(this.attachText(this.successText));
  }

  describe() {
    return `我在和 ${this.getTarget()} 使用模板 ${this.model} 玩耍。\n`;
  }
}

@RegisterSchema()
export class EdgingProfileConfig extends EdgingProfile {}

@RegisterSchema()
export class EdgingPluginConfig {
  constructor(config: EdgingPluginConfigLike) {}

  @SchemaProperty({ description: '模板文件存放路径。', default: './models' })
  modelRoot: string;

  @SchemaProperty({ description: '图片标签。', type: String })
  picTags: string[];

  @SchemaProperty({ description: '图源标签。', type: String })
  sourceTags: string[];

  @SchemaProperty({ description: '默认配置。' })
  defaultProfile: EdgingProfileConfig;

  @SchemaProperty({
    description: '允许进行配置的选择器。',
    type: 'object',
    default: {},
  })
  panelSelection: Selection;

  @SchemaProperty({ description: '最小文本间隔。', default: 1000 })
  minTextInterval: number;

  @SchemaProperty({ description: '最大文本间隔。', default: 50000 })
  maxTextInterval: number;

  @SchemaProperty({ description: '最小图片间隔。', default: 5000 })
  minPicInterval: number;

  @SchemaProperty({ description: '最大图片间隔。', default: 50000 })
  maxPicInterval: number;
}

export type EdgingPluginConfigLike = Partial<EdgingPluginConfig>;
