// import 'source-map-support/register';
import { DrawPluginConfig } from './config';
import {
  DefinePlugin,
  InjectLogger,
  LifecycleEvents,
  UseCommand,
  CommandLocale,
  PutArg,
  PutCommonRenderer,
  CRenderer,
  PutUserName,
  Renderer,
  PutRenderer,
  StarterPlugin,
} from 'koishi-thirdeye';
import { Logger, Random } from 'koishi';
import path from 'path';
import loadJsonFile from 'load-json-file';
import _ from 'lodash';
import * as localeZh from './locales/zh';
import * as localeEn from './locales/en';
import leven from 'leven';
export * from './config';

type Decks = Record<string, string[]>;

@DefinePlugin()
export default class DrawPlugin
  extends StarterPlugin(DrawPluginConfig)
  implements LifecycleEvents
{
  @InjectLogger()
  private logger: Logger;

  deckFiles = new Map<string, string[]>();
  decks: Decks = {};

  async loadDeck(file: string) {
    const filename = path.basename(file);
    if (this.deckFiles.has(filename)) return;
    try {
      const content: Decks = await loadJsonFile(file);
      const deckTitles = Object.keys(content);
      this.deckFiles.set(filename, deckTitles);
      for (const key of deckTitles) {
        if (this.decks[key]) {
          this.logger.warn(`Duplicate deck ${key} in ${file}`);
        }
        this.decks[key] = content[key];
      }
    } catch (e) {
      this.logger.error(`Load deck file ${file} failed: ${e.message}`);
    }
  }

  async loadDecks() {
    this.deckFiles.clear();
    this.decks = {};
    const files = await this.config.loadFileList();
    await Promise.all(files.map((file) => this.loadDeck(file)));
    const deckCount = _.sumBy(
      Array.from(this.deckFiles.values()),
      (v) => v.length,
    );
    const deckFileCount = this.deckFiles.size;
    this.logger.info(
      `Loaded ${deckCount} decks from ${deckFileCount} deck files.`,
    );
    return { deckCount, deckFileCount };
  }

  parseDicePattern(pattern: string) {
    if (pattern.match(/^\d+$/)) {
      return parseInt(pattern);
    }
    const match = pattern.match(/^(\d*)d(\d+)$/);
    if (!match) {
      return 0;
    }
    const [, _count, _sides] = match;
    const count = _count ? parseInt(_count) : 1;
    const sides = parseInt(_sides);
    return _.sum(_.range(count).map(() => Random.int(1, sides + 1)));
  }

  parseDice(dice: string) {
    const patterns = dice.split('+');
    return _.sumBy(patterns, (pattern) => this.parseDicePattern(pattern));
  }

  parseEntry(name: string, entry: string, depth = 1): string {
    let result = entry.replace(/\[[d\d\+]+\]/g, (dicePattern) =>
      this.parseDice(dicePattern.slice(1, -1)).toString(),
    );
    if (depth > this.config.maxDepth) {
      this.logger.warn(
        `Max depth ${this.config.maxDepth} reached in deck ${name}.`,
      );
      return entry;
    }
    const usedUniqueIndex = new Map<string, Set<number>>();
    result = result
      .replace(
        /\{[^%\}]+\}/g,
        (refPattern) =>
          this.drawFromDeck(refPattern.slice(1, -1), depth + 1, name) ||
          refPattern,
      )
      .replace(/\{%[^%\}]+\}/g, (uniqRefPattern) => {
        const refName = uniqRefPattern.slice(2, -1);
        let indexArray = usedUniqueIndex.get(refName);
        if (!indexArray) {
          indexArray = new Set();
          usedUniqueIndex.set(refName, indexArray);
        }
        const deck = this.decks[refName];
        if (!deck) {
          this.logger.warn(
            `Referenced deck ${refName} not found in deck ${name}.`,
          );
          return uniqRefPattern;
        }

        const availableIndexes = _.range(deck.length).filter(
          (index) => !indexArray.has(index),
        );
        if (availableIndexes.length === 0) {
          this.logger.warn(
            `No more unique entries left for ${refName} in deck ${name}.`,
          );
          return uniqRefPattern;
        }
        const index = Random.pick(availableIndexes);
        indexArray.add(index);

        const entry = deck[index];
        return this.parseEntry(refName, entry, depth + 1);
      });
    return result;
  }

  drawFromDeck(name: string, depth = 1, referencedDeck?: string): string {
    const deck = this.decks[name];
    if (!deck) {
      this.logger.warn(
        `${referencedDeck ? 'Referenced deck' : 'Deck'} ${name} not found${
          referencedDeck ? `in deck ${referencedDeck}` : ''
        }.`,
      );
      return null;
    }
    const entry = Random.pick(deck);
    return this.parseEntry(name, entry, depth);
  }

  async onConnect() {
    await this.loadDecks();
  }

  @UseCommand('draw <name:text>', { checkArgCount: true })
  @CommandLocale('zh', localeZh.commands.draw)
  @CommandLocale('en', localeEn.commands.draw)
  drawCommand(
    @PutArg(0) name: string,
    @PutUserName() user: string,
    @PutCommonRenderer() renderer: CRenderer,
  ) {
    const result = this.drawFromDeck(name);
    if (!result) {
      return renderer('.notFound');
    }
    return (
      renderer('.result', {
        user,
      }) +
      '\n' +
      result
    );
  }

  @UseCommand('draw.list')
  @CommandLocale('zh', localeZh.commands.list)
  @CommandLocale('en', localeEn.commands.list)
  onListCommand(@PutRenderer('.fileInfo') renderer: Renderer) {
    const entries = Array.from(this.deckFiles.entries()).map(
      ([file, titles]) => ({
        file,
        count: titles.length,
      }),
    );
    return entries.map((entry) => renderer(entry)).join('\n');
  }

  @UseCommand('draw.help')
  @CommandLocale('zh', localeZh.commands.help)
  @CommandLocale('en', localeEn.commands.help)
  onHelpCommand(@PutRenderer('.fileInfo') renderer: Renderer) {
    const entries = Array.from(this.deckFiles.entries()).map(
      ([file, titles]) => ({
        file,
        count: titles.length,
        titles,
      }),
    );
    return entries
      .map(
        (entry) =>
          renderer(entry) +
          '\n' +
          entry.titles.map((title) => `draw ${title}`).join('\n'),
      )
      .join('\n');
  }

  @UseCommand('draw.reload')
  @CommandLocale('zh', localeZh.commands.reload)
  @CommandLocale('en', localeEn.commands.reload)
  async onReloadCommand(@PutRenderer('.result') renderer: Renderer) {
    const result = await this.loadDecks();
    return renderer(result);
  }

  @UseCommand('draw.search <word:text>', { checkArgCount: true })
  @CommandLocale('zh', localeZh.commands.search)
  @CommandLocale('en', localeEn.commands.search)
  async onSearchCommand(
    @PutArg(0) word: string,
    @PutRenderer('.result') renderer: Renderer,
    @PutRenderer('.notFound') notFoundRenderer: Renderer,
  ) {
    const allDecks = _.flatten(Array.from(this.deckFiles.values())).filter(
      (d) => d.includes(word),
    );
    if (!allDecks.length) {
      return notFoundRenderer({ word });
    }
    _.sortBy(allDecks, (deck) => leven(deck, word));
    return `${renderer()}\n${allDecks.join('|')}`;
  }
}
