// import 'source-map-support/register';
import { DNSPluginConfig } from './config';
import {
  DefinePlugin,
  StarterPlugin,
  LifecycleEvents,
  UseCommand,
  CommandLocale,
  PutArg,
  PutOption,
  PutObject,
  PutRenderer,
  Renderer,
} from 'koishi-thirdeye';
export * from './config';
import dns from 'dns';
import { zh } from './locales/zh';
import { en } from './locales/en';
import { Dict } from 'koishi';

class DomainInput {
  @PutOption('rrtype', '-r <type:string>', { fallback: 'A' })
  rrtype: string;
  @PutArg(0)
  domain: string;
  @PutRenderer('.enterDomain')
  emptyResponse: Renderer;
}

@DefinePlugin({ name: 'dns', schema: DNSPluginConfig })
export default class DNSPlugin
  extends StarterPlugin(DNSPluginConfig)
  implements LifecycleEvents
{
  resolver = new dns.promises.Resolver();

  onApply() {
    if (this.config.nameservers?.length) {
      this.resolver.setServers(this.config.nameservers);
    }
  }

  stringifyResult(result: any): string {
    if (typeof result === 'string') {
      return result;
    }
    if (Array.isArray(result)) {
      return result.map((o) => this.stringifyResult(o)).join('');
    }
    const entries = Object.entries(result as Dict<string | number>);
    const longestEntry = Math.max(
      ...entries.map(
        ([key, value]) => value.toString().length + key.length + 2,
      ),
    );
    return (
      `${'='.repeat(longestEntry)}\n` +
      Object.entries(result)
        .map(([key, value]) => `${key}: ${value}`)
        .join('\n')
    );
  }

  @UseCommand('nslookup <domain>', { checkUnknown: true })
  @CommandLocale('zh', zh.nslookup)
  @CommandLocale('en', en.nslookup)
  async onLookup(@PutObject() input: DomainInput) {
    if (!input.domain) return input.emptyResponse();
    const rrtype = input.rrtype.toUpperCase();
    let output = `${input.rrtype} ${input.domain}\n`;
    try {
      const result = await this.resolver.resolve(input.domain, rrtype);
      for (const pattern of result as any[]) {
        output += `${this.stringifyResult(pattern)}\n`;
      }
    } catch (e) {
      output += e.toString();
    }
    return output;
  }
}
