import {
  CommandAlias,
  CommandDescription,
  CommandExample,
  CommandUsage,
  DefineModel,
  Inject,
  InjectLogger,
  LifecycleEvents,
  MixinModel,
  ModelField,
  Primary,
  Provide,
  PutArg,
  PutOption,
  PutSession,
  UseCommand,
  UseModel,
} from 'koishi-thirdeye';
import {
  Channel,
  Database,
  DatabaseService,
  Logger,
  Session,
  Tables,
  User,
} from 'koishi';
import { RcRuleList } from '../utility/rc-rules';
import {
  DiceModule,
  getDefaultRollFaces,
  getRcMode,
  PutChannelProfile,
  PutIsGlobal,
  PutUserProfile,
} from '../utility/utility';
import { BaseModule } from '../utility/base-module';
import { DiceProfile } from '../utility/dice-profile';

declare module 'koishi' {
  interface Channel {
    diceProfile: DiceProfile;
  }

  interface User {
    diceProfile: DiceProfile;
  }

  interface Tables {
    diceSkill: DiceSkill;
    diceUserProfile: DiceUserProfile;
  }
}

class DiceProfileBase {
  @Primary()
  @ModelField('string(64)')
  userId: string;
  @Primary()
  @ModelField('string(64)')
  channelId: string;
}

@DefineModel('diceUserProfile')
export class DiceUserProfile extends DiceProfileBase {
  @ModelField('string(64)')
  currentCharacter: string;
}

@DefineModel('diceSkill')
export class DiceSkill extends DiceProfileBase {
  @Primary()
  @ModelField({
    type: 'string',
    length: 64,
    initial: 'default',
  })
  character: string;
  @Primary()
  @ModelField('string(32)')
  skillName: string;
  @ModelField('unsigned(1)')
  value: number;
}

@Provide('diceDb')
@MixinModel('user', { diceProfile: DiceProfile })
@MixinModel('channel', { diceProfile: DiceProfile })
@UseModel(DiceSkill, DiceUserProfile)
@DiceModule()
export class DbModule extends BaseModule implements LifecycleEvents {
  @Inject(true)
  private database: Database<Tables>;

  @Inject(true)
  private model: DatabaseService;

  @InjectLogger('dicex-db')
  private logger: Logger;

  onApply() {
    this.logger.info(`Dice database module loaded.`);
  }

  private targetPattern(isGlobal: boolean) {
    return isGlobal ? '频道' : '用户';
  }

  async getCurrentCharacter(session: Session) {
    const [record] = await this.database.get(
      'diceUserProfile',
      { userId: session.userId, channelId: session.channelId },
      ['currentCharacter'],
    );
    return record?.currentCharacter || 'default';
  }

  @UseCommand('dice/char [char:string]', '设置当前角色')
  @CommandDescription({ zh: '设置能力数值', en: 'Set skill value' })
  @CommandAlias('switch')
  @CommandExample('char 幽幽子')
  async onChar(@PutSession() session: Session, @PutArg(0) name: string) {
    if (!name) {
      return `当前角色为 ${await this.getCurrentCharacter(session)}。`;
    }
    await this.database.upsert(
      'diceUserProfile',
      [
        {
          userId: session.userId,
          channelId: session.channelId,
          currentCharacter: name,
        },
      ],
      ['userId', 'channelId'],
    );
    return `已设置当前角色为 ${name}。`;
  }

  @UseCommand('dice/st <exprs:text>')
  @CommandDescription({ zh: '设置能力数值', en: 'Set skill value' })
  @CommandExample('st 潜行 50')
  async onSetSkill(@PutSession() session: Session, @PutArg(0) expr: string) {
    const sessionInfo = {
      userId: session.userId,
      channelId: session.channelId || 'priv',
      character: await this.getCurrentCharacter(session),
    };
    const exprChain = expr.split(/\s/);
    const skillsSet = new Set<string>();
    const datas: Partial<DiceSkill>[] = [];
    const errorMessages: string[] = [];
    const appendData = (skillName: string, value: number) => {
      if (!skillName) {
        errorMessages.push('技能名称不能为空');
        return;
      }
      if (value == null || value < 0) {
        errorMessages.push(`${skillName} 的数值无效。`);
        return;
      }
      if (skillsSet.has(skillName)) {
        errorMessages.push(`${skillName} 出现重复。`);
        return;
      }
      skillsSet.add(skillName);
      datas.push({
        ...sessionInfo,
        skillName,
        value,
      });
    };
    for (let i = 0; i < exprChain.length; i++) {
      const fullMatch = exprChain[i].match(/^(\D+\d{1,3})+$/);
      if (fullMatch) {
        let skillString = '',
          valueString = '';
        const pattern = exprChain[i];
        for (let j = 0; j < pattern.length; j++) {
          const char = pattern[j];
          const isNumber = char.match(/\d/);
          if (!isNumber) {
            skillString += char;
          } else {
            valueString += char;
            if (j === pattern.length - 1 || !pattern[j + 1].match(/\d/)) {
              appendData(skillString, parseInt(valueString));
              skillString = '';
              valueString = '';
            }
          }
        }
      } else {
        const nextValue = exprChain[i + 1];
        if (nextValue.match(/^\d{1,3}$/)) {
          appendData(exprChain[i], parseInt(nextValue));
          i++;
        } else {
          return '语法错误。';
        }
      }
    }
    if (errorMessages.length > 0) {
      return errorMessages.join('\n');
    }
    if (!datas.length) {
      return;
    }
    await this.database.upsert('diceSkill', datas, [
      'userId',
      'channelId',
      'character',
      'skillName',
    ]);
    return `已设置能力数值: ${datas
      .map(({ skillName, value }) => `${skillName}=${value}`)
      .join(' ')}`;
  }

  async getAllSkills(session: Session) {
    const records = await this.database.get(
      'diceSkill',
      {
        userId: session.userId,
        channelId: session.channelId || 'priv',
        character: await this.getCurrentCharacter(session),
      },
      ['skillName', 'value'],
    );
    return Object.fromEntries(
      records.map(({ skillName, value }) => [skillName, value]),
    );
  }

  async getSkillValue(session: Session, skillName: string) {
    const [skill] = await this.database.get(
      'diceSkill',
      {
        userId: session.userId,
        channelId: session.channelId || 'priv',
        character: await this.getCurrentCharacter(session),
        skillName,
      },
      ['value'],
    );
    return skill?.value;
  }

  @UseCommand('dice/rcmode')
  @CommandDescription({ zh: '设置检点规则', en: 'Set RC rule' })
  @CommandUsage(
    `默认规则为0，规则序号如下:\n\n${RcRuleList.map(
      (rule, index) => `${index}\n${rule.text}\n`,
    ).join('\n')}`,
  )
  @CommandExample('rcmode -s 1  设置当前频道的检点规则为1。')
  setRcMode(
    @PutUserProfile() user: User,
    @PutChannelProfile() channel: Channel,
    @PutOption('set', '-s <rule:integer>  设置规则', {
      description: { en: 'Set rule' },
    })
    setRule: number,
    @PutIsGlobal() isGlobal: boolean,
  ) {
    if (isGlobal && !channel) {
      return '请在频道中使用该功能。';
    }
    if (setRule == null) {
      return `当前${this.targetPattern(isGlobal)}的检点规则如下:\n\n${
        RcRuleList[getRcMode(isGlobal ? undefined : user, channel)].text
      }`;
    }
    if (setRule < 0 || setRule >= RcRuleList.length) {
      return '规则序号不合法';
    }
    (isGlobal ? channel : user).diceProfile.rcMode = setRule;
    return `已设置当前${this.targetPattern(
      isGlobal,
    )}的检点规则为 ${setRule} 。`;
  }

  @UseCommand('dice/faces [faces:number]')
  @CommandDescription({ zh: '设置默认面数', en: 'Set default face count' })
  @CommandAlias('set')
  @CommandUsage('默认面数为 6 。')
  @CommandExample('faces -s 10  设置当前频道的默认面数为10。')
  setFaces(
    @PutUserProfile() user: User,
    @PutChannelProfile() channel: Channel,
    @PutArg(0) setFaceInArg: number,
    @PutOption('set', '-s <rule:integer>  设置面数', {
      description: { en: 'Set face count' },
    })
    setFaceInOpt: number,
    @PutIsGlobal() isGlobal: boolean,
  ) {
    if (isGlobal && !channel) {
      return '请在频道中使用该功能。';
    }
    const setFace = setFaceInOpt || setFaceInArg;
    if (setFace == null) {
      return `当前${this.targetPattern(
        isGlobal,
      )}的默认掷骰面数为 ${getDefaultRollFaces(
        isGlobal ? undefined : user,
        channel,
      )} 。`;
    }
    if (setFace < 2 || setFace > this.config.maxPoint) {
      return `面数必须在 2 到 ${this.config.maxPoint} 之间。`;
    }
    (isGlobal ? channel : user).diceProfile.defaultRollFaces = setFace;
    return `已设置当前${this.targetPattern(
      isGlobal,
    )}的默认面数为 ${setFace} 。`;
  }
}
