import {
  CommandAlias,
  CommandDescription,
  CommandExample,
  CommandUsage,
  DefineModel,
  Inject,
  InjectLogger,
  LifecycleEvents,
  MixinModel,
  ModelField,
  Primary,
  Provide,
  PutArg,
  PutOption,
  PutSession,
  UseCommand,
  UseModel,
} from 'koishi-thirdeye';
import {
  Channel,
  Database,
  DatabaseService,
  Logger,
  Session,
  Tables,
  User,
} from 'koishi';
import { RcRuleList } from '../utility/rc-rules';
import {
  DiceModule,
  getDefaultRollFaces,
  getRcMode,
  PutChannelProfile,
  PutIsGlobal,
  PutUserProfile,
} from '../utility/utility';
import { BaseModule } from '../utility/base-module';
import { DiceProfile } from '../utility/dice-profile';

declare module 'koishi' {
  interface Channel {
    diceProfile: DiceProfile;
  }

  interface User {
    diceProfile: DiceProfile;
  }

  interface Tables {
    diceSkill: DiceSkill;
  }
}

@DefineModel('diceSkill')
export class DiceSkill {
  @Primary()
  @ModelField('string(64)')
  userId: string;
  @Primary()
  @ModelField('string(64)')
  channelId: string;
  @Primary()
  @ModelField('string(32)')
  skillName: string;
  @ModelField('unsigned(1)')
  value: number;
}

@Provide('diceDb')
@MixinModel('user', { diceProfile: DiceProfile })
@MixinModel('channel', { diceProfile: DiceProfile })
@UseModel(DiceSkill)
@DiceModule()
export class DbModule extends BaseModule implements LifecycleEvents {
  @Inject(true)
  private database: Database<Tables>;

  @Inject(true)
  private model: DatabaseService;

  @InjectLogger('dicex-db')
  private logger: Logger;

  onApply() {
    this.logger.info(`Dice database module loaded.`);
  }

  private targetPattern(isGlobal: boolean) {
    return isGlobal ? '频道' : '用户';
  }

  @UseCommand('dice/st <skill:string> <value:integer>')
  @CommandDescription({ zh: '设置能力数值', en: 'Set skill value' })
  @CommandExample('st 潜行 50')
  async onSetSkill(
    @PutSession() session: Session,
    @PutArg(0) skillName: string,
    @PutArg(1) value: number,
  ) {
    if (!skillName) {
      return '请输入技能名称。';
    }
    if (value == null || value < 0 || value > 100) {
      return '数值必须在 0 到 100 之间。';
    }
    await this.database.upsert(
      'diceSkill',
      [
        {
          userId: session.userId,
          channelId: session.channelId || 'priv',
          skillName,
          value,
        },
      ],
      ['userId', 'channelId', 'skillName'],
    );
    return `已设置 ${skillName} 的能力数值为 ${value}。`;
  }

  async getSkillValue(session: Session, skillName: string) {
    const [skill] = await this.database.get(
      'diceSkill',
      {
        userId: session.userId,
        channelId: session.channelId || 'priv',
        skillName,
      },
      ['value'],
    );
    return skill?.value;
  }

  @UseCommand('dice/rcmode')
  @CommandDescription({ zh: '设置检点规则', en: 'Set RC rule' })
  @CommandUsage(
    `默认规则为0，规则序号如下:\n\n${RcRuleList.map(
      (rule, index) => `${index}\n${rule.text}\n`,
    ).join('\n')}`,
  )
  @CommandExample('rcmode -s 1  设置当前频道的检点规则为1。')
  setRcMode(
    @PutUserProfile() user: User,
    @PutChannelProfile() channel: Channel,
    @PutOption('set', '-s <rule:integer>  设置规则', {
      description: { en: 'Set rule' },
    })
    setRule: number,
    @PutIsGlobal() isGlobal: boolean,
  ) {
    if (isGlobal && !channel) {
      return '请在频道中使用该功能。';
    }
    if (setRule == null) {
      return `当前${this.targetPattern(isGlobal)}的检点规则如下:\n\n${
        RcRuleList[getRcMode(isGlobal ? undefined : user, channel)].text
      }`;
    }
    if (setRule < 0 || setRule >= RcRuleList.length) {
      return '规则序号不合法';
    }
    (isGlobal ? channel : user).diceProfile.rcMode = setRule;
    return `已设置当前${this.targetPattern(
      isGlobal,
    )}的检点规则为 ${setRule} 。`;
  }

  @UseCommand('dice/faces [faces:number]')
  @CommandDescription({ zh: '设置默认面数', en: 'Set default face count' })
  @CommandAlias('set')
  @CommandUsage('默认面数为 6 。')
  @CommandExample('faces -s 10  设置当前频道的默认面数为10。')
  setFaces(
    @PutUserProfile() user: User,
    @PutChannelProfile() channel: Channel,
    @PutArg(0) setFaceInArg: number,
    @PutOption('set', '-s <rule:integer>  设置面数', {
      description: { en: 'Set face count' },
    })
    setFaceInOpt: number,
    @PutIsGlobal() isGlobal: boolean,
  ) {
    if (isGlobal && !channel) {
      return '请在频道中使用该功能。';
    }
    const setFace = setFaceInOpt || setFaceInArg;
    if (setFace == null) {
      return `当前${this.targetPattern(
        isGlobal,
      )}的默认掷骰面数为 ${getDefaultRollFaces(
        isGlobal ? undefined : user,
        channel,
      )} 。`;
    }
    if (setFace < 2 || setFace > this.config.maxPoint) {
      return `面数必须在 2 到 ${this.config.maxPoint} 之间。`;
    }
    (isGlobal ? channel : user).diceProfile.defaultRollFaces = setFace;
    return `已设置当前${this.targetPattern(
      isGlobal,
    )}的默认面数为 ${setFace} 。`;
  }
}
