import {
  CommandAlias,
  CommandDescription,
  CommandExample,
  CommandUsage,
  Inject,
  InjectLogger,
  OnApply,
  Provide,
  PutArg,
  PutChannel,
  PutOption,
  PutUser,
  UseCommand,
} from 'koishi-thirdeye';
import { Channel, Database, Logger, Model, User } from 'koishi';
import { RcRuleList } from '../utility/rc-rules';
import {
  DiceModule,
  getDefaultRollFaces,
  getRcMode,
  PutIsGlobal,
} from '../utility/utility';
import { BaseModule } from '../utility/base-module';
import { getProfileModel } from '../utility/dice-profile';

declare module 'koishi' {
  interface Channel {
    diceRcMode: number;
    diceDefaultRollFaces: number;
  }

  interface User {
    diceRcMode: number;
    diceDefaultRollFaces: number;
  }
}

@Provide('diceDb')
@DiceModule()
export class DbModule extends BaseModule implements OnApply {
  @Inject(true)
  private database: Database;

  @Inject(true)
  private model: Model;

  @InjectLogger('dicex-db')
  private logger: Logger;

  onApply() {
    const profileModel = getProfileModel();
    this.model.extend('channel', profileModel);
    this.model.extend('user', profileModel);
    this.logger.info(`Dice database module loaded.`);
  }

  private targetPattern(isGlobal: boolean) {
    return isGlobal ? '频道' : '用户';
  }

  @UseCommand('dice/rcmode')
  @CommandDescription({ zh: '设置检点规则', en: 'Set RC rule' })
  @CommandUsage(
    `默认规则为0，规则序号如下:\n\n${RcRuleList.map(
      (rule, index) => `${index}\n${rule.text}\n`,
    ).join('\n')}`,
  )
  @CommandExample('rcmode -s 1  设置当前频道的检点规则为1。')
  setRcMode(
    @PutUser(['diceRcMode']) user: User,
    @PutChannel(['diceRcMode']) channel: Channel,
    @PutOption('set', '-s <rule:integer>  设置规则', {
      description: { en: 'Set rule' },
    })
    setRule: number,
    @PutIsGlobal() isGlobal: boolean,
  ) {
    if (isGlobal && !channel) {
      return '请在频道中使用该功能。';
    }
    if (setRule == null) {
      return `当前${this.targetPattern(isGlobal)}的检点规则如下:\n\n${
        RcRuleList[getRcMode(isGlobal ? undefined : user, channel)].text
      }`;
    }
    if (setRule < 0 || setRule >= RcRuleList.length) {
      return '规则序号不合法';
    }
    (isGlobal ? channel : user).diceRcMode = setRule;
    return `已设置当前${this.targetPattern(
      isGlobal,
    )}的检点规则为 ${setRule} 。`;
  }

  @UseCommand('dice/faces [faces:number]')
  @CommandDescription({ zh: '设置默认面数', en: 'Set default face count' })
  @CommandAlias('set')
  @CommandUsage('默认面数为 6 。')
  @CommandExample('faces -s 10  设置当前频道的默认面数为10。')
  setFaces(
    @PutUser(['diceDefaultRollFaces']) user: User,
    @PutChannel(['diceDefaultRollFaces']) channel: Channel,
    @PutArg(0) setFaceInArg: number,
    @PutOption('set', '-s <rule:integer>  设置面数', {
      description: { en: 'Set face count' },
    })
    setFaceInOpt: number,
    @PutIsGlobal() isGlobal: boolean,
  ) {
    if (isGlobal && !channel) {
      return '请在频道中使用该功能。';
    }
    const setFace = setFaceInOpt || setFaceInArg;
    if (setFace == null) {
      return `当前${this.targetPattern(
        isGlobal,
      )}的默认掷骰面数为 ${getDefaultRollFaces(
        isGlobal ? undefined : user,
        channel,
      )} 。`;
    }
    if (setFace < 2 || setFace > this.config.maxPoint) {
      return `面数必须在 2 到 ${this.config.maxPoint} 之间。`;
    }
    (isGlobal ? channel : user).diceDefaultRollFaces = setFace;
    return `已设置当前${this.targetPattern(
      isGlobal,
    )}的默认面数为 ${setFace} 。`;
  }
}
