import {
  CommandAlias,
  CommandExample,
  CommandShortcut,
  PutArg,
  PutChannel,
  PutUser,
  PutUserName,
  UseCommand,
} from 'koishi-thirdeye';
import { Channel, Random, User } from 'koishi';
import { rollRegexp } from '../utility/constant';
import { DiceModule, getDefaultRollFaces } from '../utility/utility';
import { BaseModule } from '../utility/base-module';

@DiceModule()
export class RollModule extends BaseModule {
  private getDefaultFaces(user: User, channel: Channel) {}

  @UseCommand('dice/roll [expr:string]', '掷骰')
  @CommandAlias('rd')
  @CommandShortcut('掷骰', { fuzzy: true })
  @CommandExample('roll 2d6+d10')
  onRoll(
    @PutUserName(true) username: string,
    @PutArg(0) message: string,
    @PutUser(['diceDefaultRollFaces']) user: User,
    @PutChannel(['diceDefaultRollFaces']) channel: Channel,
  ) {
    if (!message) {
      message = `1d${getDefaultRollFaces(user, channel)}`;
    }

    if (!rollRegexp.test(message)) return '表达式语法错误。';

    const { maxPoint = 1 << 16, maxTimes = 64 } = this.config;

    const expressions = message.split('+');
    let hasMultiple = false;
    let output = `${username} 掷骰：${message}=`;
    let total = 0;

    for (const expr of expressions) {
      const [, dice, _times, _max] = /^((\d*)d)?(\d+)$/i.exec(expr);
      const max = +_max;
      if (!max || max > maxPoint) {
        return `点数必须在 1 到 ${maxPoint} 之间。`;
      }

      if (!dice) {
        output += max + '+';
        total += max;
        continue;
      }

      const times = +(_times || 1);
      if (!times || times > maxTimes) {
        return `次数必须在 1 到 ${maxTimes} 之间。`;
      }

      const values = [];
      for (let index = 0; index < times; index += 1) {
        const value = Random.int(max) + 1;
        values.push(value);
        total += value;
      }
      if (times > 1) hasMultiple = true;
      if (times > 1 && expressions.length > 1) {
        output += '(';
      }
      output += values.join('+');
      if (times > 1 && expressions.length > 1) {
        output += ')';
      }
      output += '+';
    }

    output = output.slice(0, -1);
    if (hasMultiple || expressions.length > 1) {
      output += '=' + total;
    }
    return output;
  }
}
