import {
  CommandExample,
  CommandUsage,
  Inject,
  InjectConfig,
  InjectLogger,
  KoishiPlugin,
  OnApply,
  OnChannel,
  Provide,
  PutChannel,
  PutOption,
  UseCommand,
} from 'koishi-thirdeye';
import { DicePluginConfig } from '../config';
import { Channel, Context, Database, Logger, Model } from 'koishi';
import { RcRuleList } from '../utility/rc-rules';

declare module 'koishi' {
  interface Channel {
    diceRcMode: number;
  }
}

@Provide('dicexDb')
@KoishiPlugin({ name: 'dicex-db', schema: DicePluginConfig })
export class DiceDbModule implements OnApply {
  constructor(private ctx: Context, config: DicePluginConfig) {}

  @InjectConfig()
  private config: DicePluginConfig;

  @Inject(true)
  private database: Database;

  @Inject(true)
  private model: Model;

  @InjectLogger('dicex-db')
  private logger: Logger;

  onApply() {
    this.model.extend('channel', {
      diceRcMode: { type: 'integer', initial: 0 },
    });
    this.logger.info(`Dice database module loaded.`);
  }

  @OnChannel()
  @UseCommand('dice/rcmode', '设置检点规则')
  @CommandUsage(
    `默认规则为0，规则序号如下:\n\n${RcRuleList.map(
      (rule, index) => `${index}\n${rule.text}\n`,
    ).join('\n')}`,
  )
  @CommandExample('rcmode -s 1  设置当前频道的检点规则为1。')
  fetchRcMode(
    @PutChannel(['diceRcMode']) channel: Channel,
    @PutOption('set', '-s <rule:integer>  设置规则') setRule: number,
  ) {
    if (setRule == null) {
      return `当前频道的检点规则如下:\n\n${
        RcRuleList[channel.diceRcMode].text
      }`;
    }
    if (setRule < 0 || setRule >= RcRuleList.length) {
      return '规则序号不合法';
    }
    channel.diceRcMode = setRule;
    return `已设置当前频道的检点规则为${setRule}。`;
  }
}
