import 'source-map-support/register';
import { Channel, Context, NextFunction, Random, Session } from 'koishi';
import { DicePluginConfig, DicePluginConfigLike } from './config';
import {
  CommandAlias,
  CommandExample,
  CommandShortcut,
  Inject,
  InjectConfig,
  KoishiPlugin,
  OnApply,
  PutArg,
  PutChannel,
  PutUserName,
  UseCommand,
  UseMiddleware,
} from 'koishi-thirdeye';
import { DiceDbModule } from './modules/db';
import { RcResult, RcRuleList } from './utility/rc-rules';

export * from './config';

declare module 'koishi' {
  interface Modules {
    dicex: typeof import('.');
  }
  // eslint-disable-next-line @typescript-eslint/no-namespace
  namespace Context {
    interface Services {
      dicexDb: DiceDbModule;
    }
  }
}

const rollRegexp = /^((\d*)d)?(\d+)(\+((\d*)d)?(\d+))*$/i;
const rcRegexp = /^[ac](\d{1,3})( +[^ ].*)?$/i;

@KoishiPlugin({ name: 'dicex', schema: DicePluginConfig })
export default class DicePlugin implements OnApply {
  constructor(private ctx: Context, config: DicePluginConfigLike) {}
  @InjectConfig()
  private config: DicePluginConfig;

  @Inject('dicexDb')
  private db: DiceDbModule;

  @UseCommand('dice', '骰子指令', { empty: true })
  // eslint-disable-next-line @typescript-eslint/no-empty-function
  diceCommand() {}

  @UseCommand('dice/roll [expr:string]', '掷骰')
  @CommandShortcut('掷骰', { fuzzy: true })
  @CommandExample('roll 2d6+d10')
  onRoll(@PutUserName(true) username: string, @PutArg(0) message = '1d6') {
    if (!rollRegexp.test(message)) return '表达式语法错误。';

    const { maxPoint = 1 << 16, maxTimes = 64 } = this.config;

    const expressions = message.split('+');
    let hasMultiple = false;
    let output = `${username} 掷骰：${message}=`;
    let total = 0;

    for (const expr of expressions) {
      const [, dice, _times, _max] = /^((\d*)d)?(\d+)$/i.exec(expr);
      const max = +_max;
      if (!max || max > maxPoint) {
        return `点数必须在 1 到 ${maxPoint} 之间。`;
      }

      if (!dice) {
        output += max + '+';
        total += max;
        continue;
      }

      const times = +(_times || 1);
      if (!times || times > maxTimes) {
        return `次数必须在 1 到 ${maxTimes} 之间。`;
      }

      const values = [];
      for (let index = 0; index < times; index += 1) {
        const value = Random.int(max) + 1;
        values.push(value);
        total += value;
      }
      if (times > 1) hasMultiple = true;
      if (times > 1 && expressions.length > 1) {
        output += '(';
      }
      output += values.join('+');
      if (times > 1 && expressions.length > 1) {
        output += ')';
      }
      output += '+';
    }

    output = output.slice(0, -1);
    if (hasMultiple || expressions.length > 1) {
      output += '=' + total;
    }
    return output;
  }

  private getRcRule(channel: Channel) {
    const index = channel?.diceRcMode || 0;
    return RcRuleList[index];
  }

  @UseCommand('dice/rc <rate:integer> [reason:string]', '检定')
  @CommandAlias('ra')
  @CommandShortcut('检定', { fuzzy: true })
  @CommandExample('rc 20 潜行')
  onRc(
    @PutUserName(true) username: string,
    @PutArg(0) rate: number,
    @PutArg(1) reason: string,
    @PutChannel(['diceRcMode']) channel: Channel,
  ) {
    if (!rate || rate < 0 || rate > 100) {
      return '成功率必须在 0 到 100 之间。';
    }
    const rule = this.getRcRule(channel);
    const value = Random.int(1, 101);
    const result = rule.check(rate, value);
    const resultText =
      result === RcResult.BigFailure
        ? '大失败！'
        : result === RcResult.Failure
        ? '失败'
        : result === RcResult.Success
        ? '成功'
        : '大成功！';
    return `${username}${
      reason ? `要${reason}，开始` : ''
    }进行检定：D100=${value}/${rate} ${resultText}`;
  }

  @UseMiddleware()
  onRollCompat(session: Session, next: NextFunction) {
    const { content, prefix } = session.parsed;
    if (!prefix || content[0] !== 'r') return next();
    const expr = content.slice(1);
    if (rollRegexp.test(expr)) {
      return session.execute({ name: 'roll', args: [expr] });
    }
    if (rcRegexp.test(expr)) {
      const matching = expr.match(rcRegexp);
      return session.execute({
        name: 'rc',
        args: [matching[1], ...(matching[2] ? [matching[2].trim()] : [])],
      });
    }
    return next();
  }

  onApply() {
    this.ctx.plugin(DiceDbModule, this.config);
  }
}
