import { ChatGPTApiPluginConfig } from './config';
import {
  DefinePlugin,
  Inject,
  InjectLogger,
  LifecycleEvents,
  PutArg,
  PutSession,
  StarterPlugin,
  UseCommand,
  UseMiddleware,
} from 'koishi-thirdeye';
export * from './config';
import type { ChatGPTAPI, ChatMessage } from 'chatgpt';
import Cache from '@koishijs/cache';
import { Logger, Next, Session } from 'koishi';
import internal from 'stream';
import { transform } from './transform';

declare module '@koishijs/cache' {
  interface Tables {
    chatgptMessages: ChatMessage;
  }
}

@DefinePlugin({ name: 'chatgpt-api' })
export default class ChatGPTApiPlugin
  extends StarterPlugin(ChatGPTApiPluginConfig)
  implements LifecycleEvents
{
  @Inject(true)
  private __cache__: Cache;

  ChatGPTAPIConstructor: typeof ChatGPTAPI;
  async onConnect() {
    if (!globalThis.fetch) {
      const { default: fetch } = await eval("import('node-fetch')");
      globalThis.fetch = fetch;
      globalThis.Headers = fetch.Headers;
      globalThis.Request = fetch.Request;
      globalThis.Response = fetch.Response;
    }
    const { ChatGPTAPI } = await eval("import('chatgpt')");
    this.ChatGPTAPIConstructor = ChatGPTAPI;
  }

  getApiInstance(session: Session) {
    const table = this.__cache__.cache('chatgptMessages');
    return new this.ChatGPTAPIConstructor({
      apiKey: this.config.apiKey,
      completionParams: {
        model: this.config.model,
        ...this.config.completionParams,
      },
      userLabel: session.username || session.userId,
      assistantLabel: session.bot.username || session.selfId,
      getMessageById: (id) => table.get(`${internal}:${id}`),
      upsertMessage: (message) =>
        table.set(`${internal}:${message.id}`, message),
      ...this.config.extras,
    });
  }

  private sessionKey(session: Session) {
    return `session:${session.platform}:${session.selfId}:${session.channelId}:${session.userId}`;
  }

  @UseCommand('chatgpt <content:text>', '与 ChatGPT 对话，也可以直接发送消息。')
  async chat(@PutSession() session: Session, @PutArg(0) content: string) {
    return this.handle(session, content);
  }

  @UseCommand('chatgpt.reset', '重置 ChatGPT 对话')
  async reset(@PutSession() session: Session) {
    await this.__cache__
      .cache('chatgptMessages')
      .delete(this.sessionKey(session));
    return 'OK';
  }

  @InjectLogger()
  private logger: Logger;

  @UseMiddleware()
  middleware(session: Session, next: Next) {
    if (
      (!this.config.prefix.length && session.content) ||
      session.parsed?.appel
    )
      return this.handle(session, session.content);
    const matchingPrefix = this.config.prefix.find((prefix) =>
      session.content.startsWith(prefix),
    );
    if (!matchingPrefix) return next();
    return this.handle(session, session.content.slice(matchingPrefix.length));
  }

  async handle(session: Session, text: string) {
    if (!text) return;
    const api = this.getApiInstance(session);
    const key = this.sessionKey(session);
    const lastMessage = await this.__cache__.cache('chatgptMessages').get(key);
    try {
      const message = await api.sendMessage(text, {
        ...(lastMessage
          ? {
              parentMessageId: lastMessage.id,
              conversationId: lastMessage.conversationId,
            }
          : {}),
        timeoutMs: 300000,
        ...this.config.sendOptions,
      });
      await this.__cache__
        .cache('chatgptMessages')
        .set(key, message, this.config.sessionTTL);
      return transform(message.text);
    } catch (e) {
      this.logger.error(e);
    }
    return 'Internal Server Error';
  }
}
