import { Context } from 'koishi';
import TargetPlugin, { UseCache } from '../src';
import { CacheKey, WithKey } from 'aragami';
import { DefinePlugin, Provide, StarterPlugin } from 'koishi-thirdeye';

declare module '@koishijs/cache' {
  interface Tables {
    aragami: any;
  }
}

describe('Test of plugin.', () => {
  let app: Context;

  beforeEach(async () => {
    app = new Context();
    app.plugin(TargetPlugin, {
      // redis: { uri: 'redis://localhost:6379' },
    });
    await app.start();
    // console.log(app.aragami);
    // console.log(app.__cache__);
  });

  class User {
    @CacheKey()
    name: string;
    age: number;
  }

  it('should pass', async () => {
    expect(app.aragami).toBeDefined();
    const user = new User();
    user.name = 'koishi';
    user.age = 18;
    await expect(app.aragami.set(user)).resolves.toBe(user);
    const getUser = await app.aragami.get(User, 'koishi');
    expect(getUser).toEqual(user);
    expect(getUser).toBeInstanceOf(User);
  });

  it('should use wrapping decorator', async () => {
    @Provide('test')
    @DefinePlugin()
    class MyPlugin extends StarterPlugin() {
      @UseCache(User)
      async getUser(@WithKey() name: string, age: number) {
        const user = new User();
        user.name = name;
        user.age = age;
        return user;
      }
    }
    app.plugin(MyPlugin);
    const instance: MyPlugin = app['test'];
    expect(instance).toBeDefined();
    await instance.getUser('Satori', 18);
    await expect(app.aragami.has(User, 'Satori')).resolves.toBeTruthy();
  });

  it('should work with builtin cache', async () => {
    expect(app.__cache__).toBeDefined();
    const table = app.cache('aragami');
    expect(table).toBeDefined();
    await table.set('foo', { bar: 'baz' }, 10);
    await expect(table.get('foo')).resolves.toEqual({ bar: 'baz' });
    await new Promise((resolve) => setTimeout(resolve, 15));
    await expect(table.get('foo')).resolves.toBeUndefined();
  });
});
