import { AragamiConfig } from './config';
import {
  DefinePlugin,
  Inject,
  koishiRegistrar,
  LifecycleEvents,
  PartialDeep,
  PluginDef,
  PluginSchema,
  Provide,
  UsePlugin,
} from 'koishi-thirdeye';
import { Channel, Command, Context, Session, User } from 'koishi';
import {
  Aragami,
  AragamiOptions,
  CacheKey,
  WrapDecoratorBuilder,
} from 'aragami';
export * from './config';
export * from 'aragami';
import Cache from '@koishijs/cache';

declare module 'koishi' {
  interface Context {
    aragami: AragamiPlugin;
  }
}

class AdaptingCache {
  table: string;
  key: string;
  value: any;

  @CacheKey()
  cacheKey() {
    return `${this.table}:${this.key}`;
  }
}

type CopyType<T> = { [P in keyof T]: T[P] };

@DefinePlugin()
export class AragamiCacheProvider extends Cache {
  constructor(ctx: Context) {
    super(ctx);
  }

  @Inject()
  private aragami: CopyType<AragamiPlugin>;

  async clear(table: string) {
    return this.aragami.clear(AdaptingCache, table);
  }

  async get(table: string, key: string) {
    const entry = await this.aragami.get(AdaptingCache, `${table}:${key}`);
    return entry?.value;
  }

  async set(table: string, key: string, value: any, maxAge?: number) {
    const entry = new AdaptingCache();
    entry.table = table;
    entry.key = key;
    entry.value = value;
    await this.aragami.set(entry, { ttl: maxAge ?? 0 });
  }

  async delete(table: string, key: string) {
    await this.aragami.del(AdaptingCache, `${table}:${key}`);
  }
}

@PluginSchema(AragamiConfig)
@Provide('aragami', { immediate: true })
@DefinePlugin({ name: 'cache-aragami' })
export default class AragamiPlugin extends Aragami implements LifecycleEvents {
  constructor(
    ctx: Context,
    _config: PartialDeep<AragamiConfig> & AragamiOptions,
  ) {
    super(_config.getConfig());
  }

  @UsePlugin()
  loadAdaptingProvider() {
    //console.log('loadAdaptingProvider');
    return PluginDef(AragamiCacheProvider);
  }

  onDisconnect() {
    return this.destroy();
  }

  private commandLock<
    U extends User.Field = never,
    G extends Channel.Field = never,
    A extends any[] = any[],
    // eslint-disable-next-line @typescript-eslint/ban-types
    O extends {} = {},
  >(
    cmd: Command<U, G, A, O>,
    fields: string[],
    info: { idFactory: (session: Session) => string; prefix: string },
  ) {
    return cmd.action((argv, ...args: any[]) => {
      if (!argv.session) return argv.next();
      const id = info.idFactory(argv.session);
      if (!id) return argv.next();
      const lockFields = fields.map(
        (field) => `koishi:${info.prefix}:${id}:${field}`,
      );
      return this.lock(lockFields, argv.next);
    }, true);
  }

  commandUserLock<
    T extends User.Field,
    U extends User.Field = never,
    G extends Channel.Field = never,
    A extends any[] = any[],
    // eslint-disable-next-line @typescript-eslint/ban-types
    O extends {} = {},
  >(cmd: Command<U, G, A, O>, fields: T[]) {
    return this.commandLock(cmd.userFields(fields), fields as string[], {
      idFactory: (session: Session) => session.userId,
      prefix: 'user',
    });
  }

  commandChannelLock<
    T extends Channel.Field,
    U extends User.Field = never,
    G extends Channel.Field = never,
    A extends any[] = any[],
    // eslint-disable-next-line @typescript-eslint/ban-types
    O extends {} = {},
  >(cmd: Command<U, G, A, O>, fields: T[]) {
    return this.commandLock(cmd.channelFields(fields), fields as string[], {
      idFactory: (session: Session) => session.channelId,
      prefix: 'channel',
    });
  }

  commandGuildLock<
    T extends Channel.Field,
    U extends User.Field = never,
    G extends Channel.Field = never,
    A extends any[] = any[],
    // eslint-disable-next-line @typescript-eslint/ban-types
    O extends {} = {},
  >(cmd: Command<U, G, A, O>, fields: T[]) {
    return this.commandLock(cmd.channelFields(fields), fields as string[], {
      idFactory: (session: Session) => session.guildId || session.channelId,
      prefix: 'guild',
    });
  }
}

export const { UseCache, UseLock } = new WrapDecoratorBuilder(
  (o) => ((o.ctx || o.__ctx) as Context).aragami,
).build();

export const CommandUserLock =
  <T extends User.Field>(...fields: T[]) =>
  <
    U extends User.Field = never,
    G extends Channel.Field = never,
    A extends any[] = any[],
    // eslint-disable-next-line @typescript-eslint/ban-types
    O extends {} = {},
  >(
    cmd: Command<U, G, A, O>,
  ) =>
    cmd.ctx.aragami.commandUserLock(cmd, fields);

export const CommandChannelLock =
  <T extends Channel.Field>(...fields: T[]) =>
  <
    U extends User.Field = never,
    G extends Channel.Field = never,
    A extends any[] = any[],
    // eslint-disable-next-line @typescript-eslint/ban-types
    O extends {} = {},
  >(
    cmd: Command<U, G, A, O>,
  ) =>
    cmd.ctx.aragami.commandChannelLock(cmd, fields);

export const CommandGuildLock =
  <T extends Channel.Field>(...fields: T[]) =>
  <
    U extends User.Field = never,
    G extends Channel.Field = never,
    A extends any[] = any[],
    // eslint-disable-next-line @typescript-eslint/ban-types
    O extends {} = {},
  >(
    cmd: Command<U, G, A, O>,
  ) =>
    cmd.ctx.aragami.commandGuildLock(cmd, fields);

export const PutLockingUser = koishiRegistrar.decorateCommandPut(
  (info, ...fields: (keyof User)[]) => info.argv.session.user,
  (info, ...fields) => info.ctx.aragami.commandUserLock(info.command, fields),
);

export const PutLockingChannel = koishiRegistrar.decorateCommandPut(
  (info, ...fields: (keyof Channel)[]) => info.argv.session.channel,
  (info, ...fields) =>
    info.ctx.aragami.commandChannelLock(info.command, fields),
);

export const PutLockingGuild = koishiRegistrar.decorateCommandPut(
  (info, ...fields: (keyof Channel)[]) => info.argv.session.guild,
  (info, ...fields) => info.ctx.aragami.commandGuildLock(info.command, fields),
);
