// import 'source-map-support/register';
import { ApiPluginConfig, ApiPluginConfigLike } from './config';
import {
  DefinePlugin,
  InjectConfig,
  OnPlatform,
  UseMiddleware,
  Post,
  KoaContext,
  PluginSchema,
} from 'koishi-thirdeye';
import { Bot, Context, getSessionId, Next, Random } from 'koishi';
import { ApiSession } from './api-session';
import { Prompt } from './def/prompt';
export * from './config';

@PluginSchema(ApiPluginConfig)
@DefinePlugin()
export default class ApiBot extends Bot {
  username = 'koishi';
  selfId = 'koishi';
  hidden = true;
  prompts = new Map<string, Prompt>();

  constructor(public ctx: Context, config: ApiPluginConfigLike) {
    super(ctx, { platform: 'api', selfId: 'koishi' });
  }

  resolvePrompt(key: string, value: string) {
    const prompt = this.prompts.get(key);
    if (prompt) {
      prompt.resolver(value);
      clearTimeout(prompt.timeout);
      this.prompts.delete(key);
      return prompt;
    }
    return;
  }

  @OnPlatform('api')
  @UseMiddleware()
  async handlePrompt(session: ApiSession, next: Next) {
    const identifier = getSessionId(session);
    const prompt = this.resolvePrompt(identifier, session.content);
    if (!prompt) {
      return next();
    }
    session.storedMessages = session.storedMessages.concat(
      await prompt.session.waitForPattern(),
    );
    return;
  }

  @InjectConfig()
  private pluginConfig: ApiPluginConfig;

  @Post('{{path}}')
  async onHttpPost(ctx: KoaContext) {
    if (this.pluginConfig.token) {
      const header = ctx.request.headers['authorization'];
      const tokenFromRequest = header?.startsWith('Bearer ')
        ? header.slice(7)
        : header;
      if (tokenFromRequest !== this.pluginConfig.token) {
        ctx.status = 401;
        ctx.body = { error: 'Invalid token' };
        return;
      }
    }
    if (typeof ctx.request.body === 'string') {
      ctx.status = 400;
      ctx.body = { error: 'Invalid body' };
      return;
    }
    const body = ctx.request.body;
    if (!body.content) {
      ctx.status = 400;
      ctx.body = { error: 'Missing content.' };
      return;
    }
    const userId = body.userId || ctx.request.ip;
    const session = new ApiSession(this, {
      userId,
      channelId: body.channelId || `private:${userId}`,
      guildId: body.guildId,
      messageId: body.messageId || Random.id(),
      content: body.content,
      platform: 'api',
      selfId: 'koishi',
      type: 'message',
      subtype: body.channelId ? 'group' : 'private',
      author: {
        userId,
        avatar: body.avatar,
        username: body.username || userId,
      },
    });
    if (!this.ctx.filter(session)) {
      ctx.status = 200;
      ctx.body = { messages: [] };
      return;
    }
    const currentPromise = this.ctx.app.parallel('message', session);
    session.currentPromise = currentPromise;
    ctx.status = 200;
    ctx.body = { messages: await session.waitForPattern() };
  }

  async sendMessage(channelId: string, content: string) {
    return [];
  }

  async sendPrivateMessage(userId: string, content: string) {
    return [];
  }
}

export interface MessageBody {
  channelId?: string;
  guildId?: string;
  messageId?: string;
  content: string;
  userId?: string;
  username?: string;
  avatar?: string;
}
