import { App } from 'koishi';
import TargetPlugin from '../src';
import request from 'supertest';

describe('Test of plugin.', () => {
  let app: App;

  beforeEach(async () => {
    app = new App();
    app.plugin(TargetPlugin, { token: 'dress' });
    await app.start();
  });

  it('should check token', () => {
    request(app._httpServer)
      .post('/api')
      .send({ userId: '111111', content: 'wwww' })
      .expect(401);
    request(app._httpServer)
      .post('/api')
      .send({ userId: '111111', content: 'wwww' })
      .set('Authorization', 'dress')
      .expect(200);
    request(app._httpServer)
      .post('/api')
      .send({ userId: '111111', content: 'wwww' })
      .set('Authorization', 'Bearer dress')
      .expect(200);
    request(app._httpServer)
      .post('/api')
      .send({ userId: '111111', content: 'wwww' })
      .set('Authorization', 'Bearer skirt')
      .expect(401);
  });

  it('should reply message', () => {
    app.command('dress').action((a) => '梦梦女装！');
    request(app._httpServer)
      .post('/api')
      .send({ userId: '111111', content: 'dress' })
      .set('Authorization', 'dress')
      .expect(200)
      .expect({
        messages: ['梦梦女装！'],
      });
  });

  it('should correctly ref userId', () => {
    app.command('dress').action((a) => `${a.session.userId} 让梦梦女装！`);
    request(app._httpServer)
      .post('/api')
      .send({ userId: '111111', content: 'dress' })
      .set('Authorization', 'dress')
      .expect(200)
      .expect({
        messages: ['111111 让梦梦女装！'],
      });
  });

  it('should correctly ref groupId', () => {
    app.command('dress').action(async (a) => {
      await a.session.send('梦梦换衣服啦！');
      return '梦梦女装啦！';
    });
    request(app._httpServer)
      .post('/api')
      .send({ userId: '111111', content: 'dress' })
      .set('Authorization', 'dress')
      .expect(200)
      .expect({
        messages: ['梦梦换衣服啦！', '梦梦女装啦！'],
      });
  });

  it('should correctly resolve prompt', async () => {
    app.command('ask').action(async (a) => {
      await a.session.send(
        '你好，我是小美女，请问能告诉我你今天穿了什么颜色的衣服吗？',
      );
      const colorOfShirt = await a.session.prompt(100000);
      await a.session.send(`那你能告诉我你今天穿了什么颜色的裙子吗？`);
      const colorOfSkirt = await a.session.prompt(100000);
      return `原来 ${a.session.userId} 今天穿的是 ${colorOfShirt} 的衣服，和 ${colorOfSkirt} 的裙子！`;
    });
    await request(app._httpServer)
      .post('/api')
      .send({ userId: '111111', content: 'ask' })
      .set('Authorization', 'dress')
      .expect(200)
      .expect({
        messages: [
          '你好，我是小美女，请问能告诉我你今天穿了什么颜色的衣服吗？',
        ],
      })
      .then();
    await request(app._httpServer)
      .post('/api')
      .send({ userId: '111112', content: 'ask' })
      .set('Authorization', 'dress')
      .expect(200)
      .expect({
        messages: [
          '你好，我是小美女，请问能告诉我你今天穿了什么颜色的衣服吗？',
        ],
      })
      .then();
    await request(app._httpServer)
      .post('/api')
      .send({ userId: '111111', content: '红色' })
      .set('Authorization', 'dress')
      .expect(200)
      .expect({
        messages: ['那你能告诉我你今天穿了什么颜色的裙子吗？'],
      })
      .then();
    await request(app._httpServer)
      .post('/api')
      .send({ userId: '111112', content: '蓝色' })
      .set('Authorization', 'dress')
      .expect(200)
      .expect({
        messages: ['那你能告诉我你今天穿了什么颜色的裙子吗？'],
      })
      .then();
    await request(app._httpServer)
      .post('/api')
      .send({ userId: '111111', content: '白色' })
      .set('Authorization', 'dress')
      .expect(200)
      .expect({
        messages: ['原来 111111 今天穿的是 红色 的衣服，和 白色 的裙子！'],
      })
      .then();
    await request(app._httpServer)
      .post('/api')
      .send({ userId: '111112', content: '黑色' })
      .set('Authorization', 'dress')
      .expect(200)
      .expect({
        messages: ['原来 111112 今天穿的是 蓝色 的衣服，和 黑色 的裙子！'],
      })
      .then();
  });

  afterEach(async () => {
    await app.stop();
    app._httpServer.close();
  });
});
