// import 'source-map-support/register';
import { ApiPluginConfig, ApiPluginConfigLike } from './config';
import {
  DefinePlugin,
  LifecycleEvents,
  Inject,
  InjectConfig,
} from 'koishi-thirdeye';
import { Adapter, Bot, Context, getSessionId, Random, Router } from 'koishi';
import { ApiSession } from './api-session';
import { Prompt } from './def/prompt';
export * from './config';

export class ApiBot extends Bot {
  username = 'koishi';
  selfId = 'koishi';
  hidden = true;
  prompts = new Map<string, Prompt>();

  resolvePrompt(key: string, value: string) {
    const prompt = this.prompts.get(key);
    if (prompt) {
      prompt.resolver(value);
      clearTimeout(prompt.timeout);
      this.prompts.delete(key);
      return prompt;
    }
    return;
  }

  constructor(public adapter: ApiAdapter, config: Bot.BaseConfig) {
    super(adapter, config);
    this.adapter.ctx
      .any()
      .platform('api')
      .self(this.selfId)
      .middleware(async (session: ApiSession, next) => {
        const identifier = getSessionId(session);
        const prompt = this.resolvePrompt(identifier, session.content);
        if (!prompt) {
          return next();
        }
        session.storedMessages = session.storedMessages.concat(
          await prompt.session.waitForPattern(),
        );
        return;
      }, true);
  }

  async sendMessage(channelId: string, content: string) {
    return [];
  }

  async sendPrivateMessage(userId: string, content: string) {
    return [];
  }
}

export interface MessageBody {
  channelId?: string;
  guildId?: string;
  messageId?: string;
  content: string;
  userId?: string;
  username?: string;
  avatar?: string;
}

@DefinePlugin({ name: 'api', schema: ApiPluginConfig })
export default class ApiAdapter extends Adapter implements LifecycleEvents {
  constructor(ctx: Context, config: ApiPluginConfigLike) {
    super(ctx, config);
  }

  @Inject('bots', true)
  private botList: Adapter.BotList;

  @Inject(true)
  private router: Router;

  @InjectConfig()
  private pluginConfig: ApiPluginConfig;

  onApply() {
    this.platform = 'api';
    this.botList.adapters.api = this;
    const bot = this.botList.create('api', {}, ApiBot);

    this.router.post(this.pluginConfig.path, async (ctx) => {
      if (this.pluginConfig.token) {
        const header = ctx.request.headers['authorization'];
        const tokenFromRequest = header?.startsWith('Bearer ')
          ? header.slice(7)
          : header;
        if (tokenFromRequest !== this.pluginConfig.token) {
          ctx.status = 401;
          ctx.body = { error: 'Invalid token' };
          return;
        }
      }
      if (typeof ctx.request.body === 'string') {
        ctx.status = 400;
        ctx.body = { error: 'Invalid body' };
        return;
      }
      const body = ctx.request.body;
      if (!body.content) {
        ctx.status = 400;
        ctx.body = { error: 'Missing content.' };
        return;
      }
      const userId = body.userId || ctx.request.ip;
      const session = new ApiSession(bot, {
        userId,
        channelId: body.channelId || `private:${userId}`,
        guildId: body.guildId,
        messageId: body.messageId || Random.id(),
        content: body.content,
        platform: 'api',
        selfId: 'koishi',
        type: 'message',
        subtype: body.channelId ? 'group' : 'private',
        author: {
          userId,
          avatar: body.avatar,
          username: body.username || userId,
        },
      });
      const currentPromise = this.ctx.app.parallel('message', session);
      session.currentPromise = currentPromise;
      ctx.status = 200;
      ctx.body = { messages: await session.waitForPattern() };
    });
  }

  async start() {}

  async stop() {}
}
