// import 'source-map-support/register';
import { RegisterSchema, SchemaProperty } from 'koishi-thirdeye';
import { Adapter, segment } from 'koishi';
import _ from 'lodash';
import { AlertDto } from './def';

@RegisterSchema()
export class ChannelTarget {
  @SchemaProperty({ description: '频道 ID', required: true })
  channelId: string;
  @SchemaProperty({ description: '群组 ID' })
  guildId: string;

  toDesc(): string | [string, string] {
    return this.guildId ? [this.channelId, this.guildId] : this.channelId;
  }
}

@RegisterSchema()
export class SendTarget {
  @SchemaProperty({
    required: true,
    description: 'bot 标识符。例如 onebot:123456789。',
  })
  bot: string;

  @SchemaProperty({
    type: String,
    description: '私聊消息发送目标。',
    default: [],
  })
  users: string[];

  @SchemaProperty({
    type: ChannelTarget,
    description: '群聊消息发送目标。',
    default: [],
  })
  channels: Partial<ChannelTarget>[];

  async send(bots: Adapter.BotList, content: string) {
    const bot = bots.get(this.bot);
    if (!bot) {
      throw new Error(`bot ${this.bot} not found`);
    }
    return _.flatten(
      await Promise.all([
        ...this.users.map((userId) =>
          bot.sendPrivateMessage
            ? bot.sendPrivateMessage(userId, content)
            : bot.sendMessage(`@${userId}`, content),
        ),
        ...(this.channels?.length
          ? [
              bot.broadcast(
                this.channels.map((c) => c.toDesc()),
                content,
              ),
            ]
          : []),
      ]),
    );
  }
}

@RegisterSchema()
export class Instance {
  @SchemaProperty({
    required: true,
    description: 'POST 的 Authorization Bearer 字段，用于标识不同路由。',
  })
  token: string;

  @SchemaProperty({
    description: '警报标题。',
    default: '警报',
  })
  title: string;

  @SchemaProperty({
    description: '解除警报的提示信息。',
    default: '警报',
  })
  resolvedMessage: string;

  formatAlert(dto: AlertDto) {
    const { alerts } = dto;
    let text = `${segment('markdown')}# ${this.title}`;

    for (const alert of alerts) {
      text += `\n\n## ${alert.labels.alertname}\n\n`;
      for (const [key, value] of Object.entries(alert.labels)) {
        if (key === 'alertname') continue;
        text += `* ${key}: \`${value}\`\n`;
      }
      text += `\n`;
      if (alert.status === 'resolved') {
        text += this.resolvedMessage;
        continue;
      }
      text += `${alert.annotations.summary}\n\n${alert.annotations.description}`;
    }
    return text;
  }

  @SchemaProperty({
    type: SendTarget,
    description: '消息发送目标。',
    default: [],
  })
  targets: Partial<SendTarget>[];

  async send(bots: Adapter.BotList, content: string) {
    return _.flatten(
      await Promise.all(
        this.targets.map((target) => target.send(bots, content)),
      ),
    );
  }
}

@RegisterSchema()
export class AlertmanagerPluginConfig {
  constructor(config: AlertmanagerPluginConfigLike) {}

  @SchemaProperty({
    description: '请求路径。',
    default: '/alertmanager',
  })
  path: string;

  @SchemaProperty({ type: Instance })
  instances: Partial<Instance>[];

  getInstance(header: string) {
    const token = header?.startsWith('Bearer ') ? header.slice(7) : header;
    return this.instances.find((instance) => instance.token === token);
  }
}

export type AlertmanagerPluginConfigLike = Partial<AlertmanagerPluginConfig>;
