import { makeArray, segment, Session, Universal } from 'koishi';
import {
  CardEvent,
  CardMenuEvent,
  WecomChatBody,
  WecomEventResponse,
  WecomLinkBody,
  WecomLocationMessageBody,
  WecomPicBody,
  WeComUser,
  WecomVideoBody,
  WecomVoiceBody,
} from './def';
import { CacheKey } from 'koishi-plugin-cache-aragami';
import type WeComBot from './index';

export function adaptUser(user: WeComUser): Universal.User {
  return {
    userId: user.userid,
    username: user.name,
    nickname: user.name,
    avatar: user.avatar,
  };
}

export function adaptSession(bot: WeComBot, input: WecomEventResponse) {
  const { body } = input;
  if (!body) {
    return;
  }
  const session: Partial<Session> = {
    selfId: bot.selfId,
    targetId: body.ToUserName,
    messageId: body.MsgId?.toString(),
    wecom: body,
    userId: body.FromUserName,
    channelId: body.FromUserName,
    timestamp: body.CreateTime,
  };
  if (body.MsgType === 'event' && !body.Event.startsWith('template_card_')) {
    session.type = `wecom/${body.Event}`;
  } else {
    switch (body.MsgType) {
      case 'text':
        const textBody = body as WecomChatBody;
        session.content = textBody.Content.toString();
        break;
      case 'image':
        const imageBody = body as WecomPicBody;
        session.content = segment.image(imageBody.PicUrl).toString();
        break;
      case 'link':
        const linkBody = body as WecomLinkBody;
        session.content = segment(
          'a',
          { href: linkBody.Url, picUrl: linkBody.PicUrl },
          linkBody.Title,
        ).toString();
        break;
      case 'video':
        const videoBody = body as WecomVideoBody;
        session.content = segment('wecom:video', {
          mediaId: videoBody.MediaId,
          thumbMediaId: videoBody.ThumbMediaId,
        }).toString();
        break;
      case 'voice':
        const voiceBody = body as WecomVoiceBody;
        session.content = segment('wecom:voice', {
          mediaId: voiceBody.MediaId,
          format: voiceBody.Format,
        }).toString();
        break;
      case 'location':
        const locationBody = body as WecomLocationMessageBody;
        session.content = segment('wecom:location', {
          latitude: locationBody.Location_X,
          longitude: locationBody.Location_Y,
          scale: locationBody.Scale,
          label: locationBody.Label,
        }).toString();
        break;
      case 'event':
        switch (body.Event) {
          case 'template_card_event':
            const cardEvent = body as unknown as CardEvent;
            session.content = segment(
              'wecom:response',
              {
                key: cardEvent.EventKey,
                id: cardEvent.TaskId,
                type: cardEvent.CardType,
                code: cardEvent.ResponseCode,
              },
              makeArray(cardEvent.SelectedItems.SelectedItem).map((item) =>
                segment(
                  'question',
                  { key: item.QuestionKey },
                  makeArray(item.OptionIds.OptionId).map((option) =>
                    segment('option', { id: option }),
                  ),
                ),
              ),
            ).toString();
            break;
          case 'template_card_menu_event':
            const cardMenuEvent = body as unknown as CardMenuEvent;
            session.content = segment('wecom:menu', {
              key: cardMenuEvent.EventKey,
              id: cardMenuEvent.TaskId,
              type: cardMenuEvent.CardType,
              code: cardMenuEvent.ResponseCode,
            }).toString();
            break;
        }
        break;
      default:
        return;
    }
    session.type = 'message';
    session.subtype = 'private';
    session.author = {
      userId: session.userId,
    };
  }
  return session;
}

export function dispatchSession(bot: WeComBot, message: WecomEventResponse) {
  const payload = adaptSession(bot, message);
  if (!payload) return;
  const session = new Session(bot, payload);
  session.wecom = message.body;
  bot.dispatch(session);
  if (
    message.body.Event &&
    [
      'click',
      'view',
      'scan',
      'scancode_waitmsg',
      'pic_sysphoto',
      'pic_photo_or_album',
      'pic_weixin',
      'location_select',
    ].includes(message.body.Event)
  ) {
    bot.handleMenuEvent(message.body.Event, session).then();
  }
}

export class WeComToken {
  @CacheKey()
  selfId: string;
  token: string;
}

export function transformKey(
  obj: Record<string, any>,
  transformer: (key: string) => string,
  visited = new Set(),
) {
  if (typeof obj !== 'object' || visited.has(obj)) return obj;
  const result: Record<string, any> = {};
  if (visited.has(obj)) return;
  visited.add(obj);
  for (const key of Object.keys(obj)) {
    let value = obj[key];
    if (Array.isArray(value)) {
      value = value.map((item) => transformKey(item, transformer, visited));
    } else {
      value = transformKey(value, transformer, visited);
    }
    result[transformer(key)] = value;
  }
  return result;
}
