import {
  Bot,
  Fragment,
  Logger,
  makeArray,
  MaybeArray,
  Quester,
  SendOptions,
  Session,
  Universal,
} from 'koishi';
import {
  adaptMenu,
  TokenReturnMessage,
  WeComAgentInfo,
  WecomEventBody,
  WecomEvents,
  WecomMenuDef,
  WecomSendMessageResponse,
  WeComUser,
} from './def';
import {
  DefinePlugin,
  Inject,
  InjectLogger,
  PluginDef,
  PluginSchema,
  RegisterSchema,
  Reusable,
  SchemaProperty,
  UsePlugin,
} from 'koishi-thirdeye';
import Aragami from 'koishi-plugin-cache-aragami';
import { adaptUser, WeComToken } from './utils';
import { WeComAdapter } from './adapter';
import { WeComMessenger } from './message';

declare global {
  // eslint-disable-next-line @typescript-eslint/no-namespace
  namespace Satori {
    interface Session {
      wecom?: WecomEventBody;
    }
  }
}

declare module 'koishi' {
  // eslint-disable-next-line @typescript-eslint/no-empty-interface
  interface Events extends WecomEvents {}
}

@RegisterSchema()
export class WeComBotConfig {
  @SchemaProperty({
    description: '企业 ID。',
    required: true,
  })
  corpId: string;
  @SchemaProperty({
    description: '企业应用 ID。',
    required: true,
  })
  agentId: string;
  @SchemaProperty({
    description: '企业应用密钥。',
    role: 'secret',
    required: true,
  })
  secret: string;

  @SchemaProperty({
    description: '回调路径。',
    default: '/wecom',
  })
  path: string;
  @SchemaProperty({
    description: '应用消息上报 token。',
    role: 'secret',
    required: true,
  })
  token: string;
  @SchemaProperty({
    description: '应用消息上播 AES 密钥。',
    role: 'secret',
    required: true,
  })
  encodingAESKey: string;

  @SchemaProperty({
    description: '企业微信菜单配置。',
    type: WecomMenuDef,
    default: [],
  })
  menus?: WecomMenuDef[];

  platform = 'wecom';
  get selfId() {
    return `${this.corpId}:${this.agentId}`;
  }
}

@Reusable()
@PluginSchema(WeComBotConfig)
@DefinePlugin()
export default class WeComBot extends Bot<Partial<WeComBotConfig>> {
  internal = {};

  @Inject(true)
  private aragami: Aragami;

  @Inject(true)
  http: Quester;

  @InjectLogger()
  logger: Logger;

  private buttonKeyMap = new Map<string, string>();

  @UsePlugin()
  private loadAdapter() {
    return PluginDef(WeComAdapter, this);
  }

  private async initializeMenu() {
    if (!this.config.menus?.length) {
      const data = await this.http.get<WecomSendMessageResponse>(
        'https://qyapi.weixin.qq.com/cgi-bin/menu/delete',
        {
          params: {
            access_token: await this.getToken(),
            agentid: this.config.agentId,
          },
        },
      );
      if (data.errcode) {
        this.logger.error(`Failed to remove menu: ${data.errmsg}`);
        return false;
      }
      return true;
    }
    const button = this.config.menus.map((menu) =>
      adaptMenu(menu, this.buttonKeyMap),
    );
    const buttonData = { button };
    const data = await this.http.post<WecomSendMessageResponse>(
      'https://qyapi.weixin.qq.com/cgi-bin/menu/create',
      buttonData,
      {
        params: {
          access_token: await this.getToken(),
          agentid: this.config.agentId,
        },
      },
    );
    if (data.errcode) {
      this.logger.error(
        `Failed to initialize menu ${JSON.stringify(buttonData)}: ${
          data.errmsg
        }`,
      );
      return false;
    }
    return true;
  }

  private async fetchNewToken(): Promise<string> {
    try {
      const data = await this.http.get<TokenReturnMessage>(
        'https://qyapi.weixin.qq.com/cgi-bin/gettoken',
        {
          params: {
            corpid: this.config.corpId,
            corpsecret: this.config.secret,
          },
        },
      );
      if (data.errcode) {
        this.logger.error(
          `Failed to fetch secret with code ${data.errcode}: ${data.errmsg}`,
        );
        return;
      }

      const tokenObject = new WeComToken();
      tokenObject.token = data.access_token;
      tokenObject.selfId = this.selfId;
      await this.aragami.set(tokenObject, { ttl: data.expires_in * 1000 });
      return data.access_token;
    } catch (e) {
      this.logger.error(`Failed to fetch secret: ${e.toString()}`);
      return;
    }
  }

  async getToken() {
    // return this.aragami.lock(this.selfId, async () => {
    const tokenObject = await this.aragami.get(WeComToken, this.selfId);
    if (tokenObject) {
      return tokenObject.token;
    }
    return this.fetchNewToken();
    // });
  }

  async handleMenuEvent(event: string, session: Session) {
    // if (!this.app.isActive) return;
    const eventKey = session.wecom['EventKey'] as string;
    const command = this.buttonKeyMap.get(eventKey);
    if (command) {
      const result = await session.execute(command);
      if (result) {
        await this.sendPrivateMessage(session.userId, result);
      }
    }
  }

  async initialize() {
    try {
      const [self, menuResult] = await Promise.all([
        this.getSelf(),
        this.initializeMenu(),
      ]);
      if (!self) {
        this.offline(new Error('Invalid credentials.'));
        return;
      }
      if (!menuResult) {
        this.offline(new Error('Failed to initialize menu.'));
        return;
      }
      Object.assign(this, self);
      this.online();
    } catch (e) {
      this.offline(e);
    }
  }

  async getSelf(): Promise<Universal.User> {
    const token = await this.getToken();
    if (!token) {
      return;
    }
    const data = await this.http.get<WeComAgentInfo>(
      'https://qyapi.weixin.qq.com/cgi-bin/agent/list',
      {
        params: {
          access_token: token,
        },
      },
    );
    if (data.errcode) {
      this.logger.error(`Failed to get self: ${data.errmsg}`);
      return;
    }
    const agent = data.agentlist.find(
      (a) => a.agentid.toString() === this.config.agentId,
    );
    if (!agent) {
      // special things
      return { userId: this.config.selfId };
    }
    const self: Universal.User = {
      userId: this.config.selfId,
      username: agent.name,
      avatar: agent.square_logo_url,
    };
    return self;
  }

  async getUser(userId: string) {
    const data = await this.http.get<WeComUser>(
      'https://qyapi.weixin.qq.com/cgi-bin/user/get',
      {
        params: {
          access_token: await this.getToken(),
          userid: userId,
        },
      },
    );
    if (data.errcode) {
      this.logger.error(`Failed to get user ${userId}: ${data.errmsg}`);
      return;
    }
    return adaptUser(data);
  }

  async getFriendList() {
    return [];
  }

  async deleteFriend(userId: string) {}

  // guild
  async getGuild(guildId: string) {
    return undefined;
  }
  async getGuildList() {
    return [];
  }

  // guild member
  async getGuildMember(guildId: string, userId: string) {
    return undefined;
  }
  async getGuildMemberList(guildId: string) {
    return [];
  }

  // channel
  async getChannel(channelId: string, guildId?: string) {
    return undefined;
  }
  async getChannelList(guildId: string) {
    return [];
  }

  // request
  async handleFriendRequest(
    messageId: string,
    approve: boolean,
    comment?: string,
  ) {}
  async handleGuildRequest(
    messageId: string,
    approve: boolean,
    comment?: string,
  ) {}
  async handleGuildMemberRequest(
    messageId: string,
    approve: boolean,
    comment?: string,
  ) {}

  async deleteMessage(channelId: string, messageId: string) {
    const token = await this.getToken();
    if (!token) {
      this.logger.error(`Missing token.`);
      return;
    }
    try {
      const data = await this.http.post<WecomSendMessageResponse>(
        'https://qyapi.weixin.qq.com/cgi-bin/message/recall',
        { msgid: messageId },
        { params: { access_token: token } },
      );
      if (data.errcode) {
        this.logger.error(
          `Failed to delete message ${messageId}: ${data.errmsg}`,
        );
      }
    } catch (e) {
      this.logger.error(
        `Errored to delete message ${messageId}: ${e.toString()}`,
      );
    }
  }

  async sendMessage(
    channelId: string,
    content: Fragment,
    guildId?: string,
    options?: SendOptions,
  ) {
    return this.sendPrivateMessage(channelId, content, options);
  }

  async sendPrivateMessage(
    userIds: MaybeArray<string>,
    content: Fragment,
    options?: SendOptions,
  ) {
    const userIdList = makeArray(userIds);
    return new WeComMessenger(
      this,
      userIdList.join('|'),
      undefined,
      options,
    ).send(content);
  }

  async broadcast(
    channels: (string | [string, string])[],
    content: string,
    delay = this.ctx.root.config.delay.broadcast,
  ) {
    const userIds = channels.map((c) => (typeof c === 'string' ? c : c[0]));
    if (!userIds.length) return [];
    return this.sendPrivateMessage(userIds, content);
  }
}
