import { App, segment } from 'koishi';
import ExtrasInDev from './extras';
import WeComBot from '../src';
import * as Help from '@koishijs/plugin-help';
import * as fs from 'fs';
import AragamiPlugin from 'koishi-plugin-cache-aragami';

const app = new App({
  port: 3000,
  host: '0.0.0.0',
  prefix: '.',
});

app.plugin(Help);

// Some extras
app.plugin(ExtrasInDev);
app.plugin(AragamiPlugin);

// Target plugin
app.plugin(WeComBot, {
  corpId: 'corpId', // 企业 ID，在企业信息 https://work.weixin.qq.com/wework_admin/frame#profile 中查看
  agentId: 'agentId', // 应用 ID，在应用管理中查看
  secret: 'secret', // 应用密钥，在应用管理中查看
  menus: [
    {
      type: 'view', // 打开网页
      name: 'MyCard', // 按钮名称
      url: 'https://mycard.moe', // 网页地址
    },
    {
      type: 'click', // 运行指令
      name: '看图', // 按钮名称
      command: 'media -c 10000000', // 指令
    },
    {
      type: 'parent', // 子指令
      name: '更多', // 按钮名称
      children: [
        { type: 'click', name: '梦梦女装', command: 'markdown' },
        { type: 'scancode_push', name: '扫一扫' }, // 扫一扫并打开网页。
        { type: 'scancode_waitmsg', name: '扫一扫2', command: 'scan' }, // 扫一扫并运行指令。
        { type: 'location_select', name: '地理位置', command: 'location' }, // 选择地理位置并运行指令。
        { type: 'click', name: '点菜', command: 'dish' }, // 点菜。
      ],
    },
  ],
  path: '/wecom', // 回调 API 路径
  token: 'token', // 回调 API token，在应用消息接收设置中设置并填入
  encodingAESKey: 'encodingAESKey', // 加密密钥，在应用消息接收设置中设置并填入
});

app.on('wecom/LOCATION', (session) => {
  console.log(
    `Got location for ${session.userId}: ${session.wecom.Longitude} ${session.wecom.Latitude}`,
  );
});

app.on('wecom/enter_agent', (session) => {
  console.log(`Got enter_agent for ${session.userId}`);
});

app.on('wecom/click', (session) => {
  console.log(`${session.userId} clicked ${session.wecom.EventKey}`);
});

app.on('message', (session) => {
  console.log(`Got message ${session.content}`);
});

app
  .command('image')
  .action(async () =>
    segment.image(await fs.promises.readFile(__dirname + '/10000.jpg')),
  );

app.command('markdown').action(async () => {
  return segment('message', { markdown: 1 }, ['# 女装\n> 今天梦梦女装了吗？']);
});

app
  .command('scan')
  .action(
    (argv) => `扫出来了: ${argv.session.wecom['ScanCodeInfo']['ScanResult']}`,
  );

app.command('location').action(async (argv) => {
  const { session } = argv;
  return `你的位置是 ${session.wecom['SendLocationInfo']['Label']}，经纬度是 ${session.wecom['SendLocationInfo']['Location_X']} ${session.wecom['SendLocationInfo']['Location_Y']}`;
});

app
  .command('media')
  .option('type', '-t <type:string>', { fallback: 'image' })
  .option('cardcode', '-c <cardcode:string>', { fallback: '10000' })
  .action(async (argv) =>
    segment(argv.options.type, {
      url: `https://cdn02.moecube.com:444/images/ygopro-images-zh-CN/${argv.options.cardcode}.jpg`,
    }),
  );

app.command('dish').action(async (argv) => {
  await argv.session.send(
    segment(
      'wecom:card',
      {
        card_type: 'multiple_interaction',
      },
      [
        segment('main_title', {
          title: '晚上想吃什么？',
          desc: '有很多好吃的呢！',
        }),
        segment('select_list', {}, [
          segment(
            '',
            {
              question_key: 'main',
              title: '主食',
            },
            [
              segment('', { id: 'rice', text: '米饭' }),
              segment('', { id: 'noodle', text: '面条' }),
            ],
          ),
          segment(
            '',
            {
              question_key: 'meat',
              title: '肉类',
            },
            [
              segment('', { id: 'beef', text: '牛肉' }),
              segment('', { id: 'pork', text: '猪肉' }),
              segment('', { id: 'chicken', text: '鸡肉' }),
            ],
          ),
          segment(
            '',
            {
              question_key: 'soup',
              title: '汤',
            },
            [
              segment('option_list', {}, [
                segment('', { id: 'tomato', text: '番茄汤' }),
                segment('', { id: 'mushroom', text: '蘑菇汤' }),
                segment('', { id: 'fish', text: '鱼汤' }),
              ]),
            ],
          ),
        ]),
        segment('submit_button', {
          text: '点菜！',
          key: 'submit',
        }),
      ],
    ),
  );
  const result = await argv.session.prompt();
  console.log(result?.toString());
  const elem = segment.normalize(result)[0];
  const mainDish = elem.children.find((e) => e.attrs.key === 'main').children[0]
    .attrs.id;
  const meat = elem.children.find((e) => e.attrs.key === 'meat').children[0]
    .attrs.id;
  const soup = elem.children.find((e) => e.attrs.key === 'soup').children[0]
    .attrs.id;
  return `你点的是：${mainDish}，${meat}，${soup}`;
});

app.on('bot-status-updated', (bot) => {
  console.log(`Bot ${bot.selfId} status updated: ${bot.status}`);
});

app.start();
