import {
  Adapter,
  Bot,
  Dict,
  omit,
  Quester,
  Schema,
  segment,
  Session,
} from 'koishi';
import moment, { Moment } from 'moment';
import {
  CommonOutMessage,
  ImageOutMessage,
  OutMessage,
  TextOutMessage,
  TokenReturnMessage,
  VideoOutMessage,
  WecomMediaUploadResponse,
  WecomSendMessageResponse,
  WeComUser,
  WeComAgentInfo,
  WecomMenuDef,
  adaptMenu,
} from './def';
import { AdapterConfig, adaptUser } from './utils';
import FormData from 'form-data';
import * as fs from 'fs';
import path from 'path';
import FileType from 'file-type';

export interface BotConfig extends Bot.BaseConfig, Quester.Config {
  corpId?: string;
  agentId?: string;
  secret?: string;
  menus?: WecomMenuDef[];
}

export const BotConfig: Schema<BotConfig> = Schema.object({
  corpId: Schema.string().required().description('企业 ID。'),
  agentId: Schema.string().required().description('企业应用 ID。'),
  secret: Schema.string()
    .role('secret')
    .required()
    .description('企业应用密钥。'),
  menus: Schema.array(WecomMenuDef).description('企业微信菜单项。'),
  ...omit(Quester.Config.dict, ['endpoint']),
});

export class WeComBot extends Bot<BotConfig> {
  static schema = AdapterConfig;

  http: Quester;
  private accessToken: string;
  private accessTokenUntil: Moment;
  private buttonKeyMap = new Map<string, string>();

  constructor(adapter: Adapter, config: BotConfig) {
    super(adapter, config);
    this.http = adapter.ctx.http.extend(config as Quester.Config);
    this.selfId = config.agentId;
  }

  private async initializeMenu() {
    if (!this.config.menus?.length) {
      const data = await this.http.get<WecomSendMessageResponse>(
        'https://qyapi.weixin.qq.com/cgi-bin/menu/delete',
        {
          params: {
            access_token: await this.getToken(),
            agentid: this.config.agentId,
          },
        },
      );
      if (data.errcode) {
        this.logger.error(`Failed to remove menu: ${data.errmsg}`);
        return false;
      }
      return true;
    }
    const button = this.config.menus.map((menu) =>
      adaptMenu(menu, this.buttonKeyMap),
    );
    const buttonData = { button };
    const data = await this.http.post<WecomSendMessageResponse>(
      'https://qyapi.weixin.qq.com/cgi-bin/menu/create',
      buttonData,
      {
        params: {
          access_token: await this.getToken(),
          agentid: this.config.agentId,
        },
      },
    );
    if (data.errcode) {
      this.logger.error(
        `Failed to initialize menu ${JSON.stringify(buttonData)}: ${
          data.errmsg
        }`,
      );
      return false;
    }
    return true;
  }

  async handleMenuEvent(event: string, session: Session) {
    // if (!this.app.isActive) return;
    const eventKey = session.wecom['EventKey'] as string;
    const command = this.buttonKeyMap.get(eventKey);
    if (command) {
      const result = await session.execute(command);
      if (result) {
        await this.sendPrivateMessage(session.userId, result);
      }
    }
  }

  async initialize() {
    try {
      const [self, menuResult] = await Promise.all([
        this.getSelf(),
        this.initializeMenu(),
      ]);
      if (!self) {
        this.reject(new Error('Invalid credentials.'));
        return;
      }
      if (!menuResult) {
        this.reject(new Error('Failed to initialize menu.'));
        return;
      }
      Object.assign(this, self);
      this.resolve();
    } catch (e) {
      this.reject(e);
    }
  }

  private async fetchNewToken(): Promise<string> {
    try {
      const data = await this.http.get<TokenReturnMessage>(
        'https://qyapi.weixin.qq.com/cgi-bin/gettoken',
        {
          params: {
            corpid: this.config.corpId,
            corpsecret: this.config.secret,
          },
        },
      );
      if (data.errcode) {
        this.logger.error(
          `Failed to fetch secret with code ${data.errcode}: ${data.errmsg}`,
        );
        return;
      }

      this.accessToken = data.access_token;
      this.accessTokenUntil = moment().add(data.expires_in, 'seconds');
      return data.access_token;
    } catch (e) {
      this.logger.error(`Failed to fetch secret: ${e.toString()}`);
      return;
    }
  }

  async getToken() {
    if (this.accessToken && moment().isBefore(this.accessTokenUntil)) {
      return this.accessToken;
    }
    return this.fetchNewToken();
  }

  async getSelf(): Promise<Bot.User> {
    const token = await this.getToken();
    if (!token) {
      return;
    }
    const data = await this.http.get<WeComAgentInfo>(
      'https://qyapi.weixin.qq.com/cgi-bin/agent/list',
      {
        params: {
          access_token: token,
        },
      },
    );
    if (data.errcode) {
      this.logger.error(`Failed to get self: ${data.errmsg}`);
      return;
    }
    const agent = data.agentlist.find(
      (a) => a.agentid.toString() === this.selfId,
    );
    if (!agent) {
      // special things
      return { userId: this.selfId };
    }
    const self: Bot.User = {
      userId: agent.agentid.toString(),
      username: agent.name,
      avatar: agent.square_logo_url,
    };
    return self;
  }
  async getUser(userId: string) {
    const data = await this.http.get<WeComUser>(
      'https://qyapi.weixin.qq.com/cgi-bin/user/get',
      {
        params: {
          access_token: await this.getToken(),
          userid: userId,
        },
      },
    );
    if (data.errcode) {
      this.logger.error(`Failed to get user ${userId}: ${data.errmsg}`);
      return;
    }
    return adaptUser(data);
  }

  async getFriendList() {
    return [];
  }

  async deleteFriend(userId: string) {}

  // guild
  async getGuild(guildId: string) {
    return undefined;
  }
  async getGuildList() {
    return [];
  }

  // guild member
  async getGuildMember(guildId: string, userId: string) {
    return undefined;
  }
  async getGuildMemberList(guildId: string) {
    return [];
  }

  // channel
  async getChannel(channelId: string, guildId?: string) {
    return undefined;
  }
  async getChannelList(guildId: string) {
    return [];
  }

  // request
  async handleFriendRequest(
    messageId: string,
    approve: boolean,
    comment?: string,
  ) {}
  async handleGuildRequest(
    messageId: string,
    approve: boolean,
    comment?: string,
  ) {}
  async handleGuildMemberRequest(
    messageId: string,
    approve: boolean,
    comment?: string,
  ) {}

  async sendGenericMessage(messageInfo: OutMessage): Promise<string> {
    const token = await this.getToken();
    if (!token) {
      this.logger.error(`Missing token.`);
      return undefined;
    }
    try {
      const data = await this.http.post<WecomSendMessageResponse>(
        'https://qyapi.weixin.qq.com/cgi-bin/message/send',
        messageInfo,
        { params: { access_token: token } },
      );
      if (data.errcode) {
        this.logger.error(
          `Failed to send message ${JSON.stringify(messageInfo)}: ${
            data.errmsg
          }`,
        );
      }
      if (data.invaliduser) {
        this.logger.error(`Invalid users: ${data.invaliduser}`);
      }
      if (data.invalidparty) {
        this.logger.error(`Invalid parties: ${data.invalidparty}`);
      }
      if (data.invalidtag) {
        this.logger.error(`Invalid tags: ${data.invalidtag}`);
      }
      return data.msgid;
    } catch (e) {
      this.logger.error(
        `Failed to send message ${JSON.stringify(
          messageInfo,
        )}: ${e.toString()}`,
      );
    }
    return undefined;
  }

  async uploadMedia(
    content: Buffer,
    type = 'image',
    fileName?: string,
  ): Promise<string> {
    const token = await this.getToken();
    if (!token) {
      this.logger.error(`Missing token.`);
      return undefined;
    }
    const form = new FormData();
    form.append('media', content, fileName);
    try {
      const data = await this.http.post<WecomMediaUploadResponse>(
        'https://qyapi.weixin.qq.com/cgi-bin/media/upload',
        form,
        { params: { access_token: token, type }, headers: form.getHeaders() },
      );
      if (data.errcode) {
        this.logger.error(`Failed to upload media ${fileName}: ${data.errmsg}`);
      }
      return data.media_id;
    } catch (e) {
      this.logger.error(`Failed to upload media ${fileName}: ${e.toString()}`);
    }
  }

  async sendMarkdownMessage(
    message: string,
    targetUsers: string[],
    extras: any = {},
  ) {
    const messageInfo: CommonOutMessage = {
      agentid: this.selfId,
      msgtype: 'markdown',
      markdown: { content: message },
      touser: targetUsers.join('|'),
      ...extras,
    };
    return this.sendGenericMessage(messageInfo);
  }

  async sendTextMessage(
    message: string,
    targetUsers: string[],
    extras: any = {},
  ) {
    const messageInfo: TextOutMessage = {
      agentid: this.selfId,
      msgtype: 'text',
      text: { content: message },
      touser: targetUsers.join('|'),
      ...extras,
    };
    return this.sendGenericMessage(messageInfo);
  }

  async sendMediaMessage(
    type: string,
    fileName: string,
    message: Buffer,
    targetUsers: string[],
    extras: any = {},
  ) {
    const mediaId = await this.uploadMedia(message, type, fileName);
    if (!mediaId) {
      return;
    }
    const messageInfo: ImageOutMessage | VideoOutMessage = {
      agentid: this.selfId,
      msgtype: type,
      [type]: { media_id: mediaId },
      touser: targetUsers.join('|'),
      ...extras,
    };
    return this.sendGenericMessage(messageInfo);
  }

  async prepareBufferAndFilename(type: string, data: Dict<string>) {
    const { url } = data;
    if (!url) {
      return;
    }
    let buffer: Buffer;
    if (url.startsWith('file://')) {
      buffer = await fs.promises.readFile(url.slice(7));
    } else if (url.startsWith('base64://')) {
      buffer = Buffer.from(data.url.slice(9), 'base64');
    } else {
      buffer = await this.http.get(url, {
        responseType: 'arraybuffer',
        headers: { accept: type + '/*' },
      });
    }
    let filename = data.file;
    if (!filename) {
      if (!url.startsWith('base64://')) {
        filename = path.basename(url);
      } else {
        const fileType = await FileType.fromBuffer(buffer);
        if (fileType) {
          filename = `media.${fileType.ext}`;
        } else {
          filename = 'media.bin';
        }
      }
    }
    return { buffer, filename };
  }

  async sendMessage(channelId: string, content: string, guildId?: string) {
    return this.sendPrivateMessage(channelId, content);
  }

  async sendPrivateMessage(
    userId: string,
    content: string,
    extraUsers: string[] = [],
  ) {
    const session = await this.session({
      content,
      subtype: 'private',
      userId,
      channelId: userId,
    });
    if (!session?.content) return [];
    const userIds = [userId, ...extraUsers];
    const chain = segment.parse(session.content);
    const messageIds: string[] = [];
    let isMarkdown = false;
    for (const code of chain) {
      const { type, data } = code;
      try {
        switch (type) {
          case 'markdown':
            isMarkdown = true;
            break;
          case 'text':
            const content = data.content.trim();
            messageIds.push(
              isMarkdown
                ? await this.sendMarkdownMessage(content, userIds)
                : await this.sendTextMessage(content, userIds),
            );
            break;
          case 'image':
          case 'video':
          case 'file':
          case 'voice':
            if (!data.url) {
              break;
            }
            const { buffer, filename } = await this.prepareBufferAndFilename(
              type,
              data,
            );
            const mediaId = await this.uploadMedia(buffer, type, filename);
            if (!mediaId) {
              break;
            }
            messageIds.push(
              await this.sendMediaMessage(type, filename, buffer, userIds),
            );
            break;
          default:
            break;
        }
      } catch (e) {
        this.logger.error(
          `Failed to send part ${type} ${data}: ${e.toString()}`,
        );
      }
    }
    return messageIds;
  }

  async broadcast(
    channels: (string | [string, string])[],
    content: string,
    delay = this.app.options.delay.broadcast,
  ) {
    const userIds = channels.map((c) => (typeof c === 'string' ? c : c[0]));
    if (!userIds.length) return [];
    return this.sendPrivateMessage(userIds[0], content, userIds.slice(1));
  }

  async deleteMessage(channelId: string, messageId: string) {
    const token = await this.getToken();
    if (!token) {
      this.logger.error(`Missing token.`);
      return;
    }
    try {
      const data = await this.http.post<WecomSendMessageResponse>(
        'https://qyapi.weixin.qq.com/cgi-bin/message/recall',
        { msgid: messageId },
        { params: { access_token: token } },
      );
      if (data.errcode) {
        this.logger.error(
          `Failed to delete message ${messageId}: ${data.errmsg}`,
        );
      }
    } catch (e) {
      this.logger.error(
        `Errored to delete message ${messageId}: ${e.toString()}`,
      );
    }
  }
}
