import { Dict, Element, Messenger, segment, snakeCase } from 'koishi';
import type WeComBot from './index';
import {
  CommonOutMessage,
  ImageOutMessage,
  OutMessage,
  TextOutMessage,
  VideoOutMessage,
  WecomMediaUploadResponse,
  WecomSendMessageResponse,
} from './def';
import FormData from 'form-data';
import fs from 'fs';
import path from 'path';
import FileType from 'file-type';
import cryptoRandomString from 'crypto-random-string';
import { transformKey } from './utils';

export class WeComMessenger extends Messenger<WeComBot> {
  private buffer = '';

  async sendGenericMessage(messageInfo: OutMessage): Promise<string> {
    const token = await this.bot.getToken();
    if (!token) {
      this.bot.logger.error(`Missing token.`);
      return undefined;
    }
    const payload = {
      agentid: this.bot.config.agentId,
      touser: this.channelId,
      ...messageInfo,
    };
    const data = await this.bot.http.post<WecomSendMessageResponse>(
      'https://qyapi.weixin.qq.com/cgi-bin/message/send',
      payload,
      { params: { access_token: token } },
    );
    if (data.errcode) {
      this.bot.logger.error(
        `Failed to send message ${JSON.stringify(payload)}: ${data.errmsg}`,
      );
    }
    if (data.invaliduser) {
      this.bot.logger.error(`Invalid users: ${data.invaliduser}`);
    }
    if (data.invalidparty) {
      this.bot.logger.error(`Invalid parties: ${data.invalidparty}`);
    }
    if (data.invalidtag) {
      this.bot.logger.error(`Invalid tags: ${data.invalidtag}`);
    }
    return data.msgid;
  }

  async uploadMedia(
    content: Buffer,
    type = 'image',
    fileName?: string,
  ): Promise<string> {
    const token = await this.bot.getToken();
    if (!token) {
      this.bot.logger.error(`Missing token.`);
      return undefined;
    }
    const form = new FormData();
    form.append('media', content, fileName);
    const data = await this.bot.http.post<WecomMediaUploadResponse>(
      'https://qyapi.weixin.qq.com/cgi-bin/media/upload',
      form,
      { params: { access_token: token, type }, headers: form.getHeaders() },
    );
    if (data.errcode) {
      this.bot.logger.error(
        `Failed to upload media ${fileName}: ${data.errmsg}`,
      );
    }
    return data.media_id;
  }

  async sendMarkdownMessage(message: string, extras: any = {}) {
    const messageInfo: CommonOutMessage = {
      msgtype: 'markdown',
      markdown: { content: message },
      ...extras,
    };
    return this.sendGenericMessage(messageInfo);
  }

  async sendTextMessage(message: string, extras: any = {}) {
    const messageInfo: TextOutMessage = {
      msgtype: 'text',
      text: { content: message },
      ...extras,
    };
    return this.sendGenericMessage(messageInfo);
  }

  async sendMediaMessage(
    type: string,
    fileName: string,
    message: Buffer,
    extras: any = {},
  ) {
    const mediaId = await this.uploadMedia(message, type, fileName);
    if (!mediaId) {
      return;
    }
    const messageInfo: ImageOutMessage | VideoOutMessage = {
      msgtype: type,
      [type]: { media_id: mediaId },
      ...extras,
    };
    return this.sendGenericMessage(messageInfo);
  }

  async prepareBufferAndFilename(type: string, data: Dict<string>) {
    const { url } = data;
    if (!url) {
      return;
    }

    if (url.startsWith('file://')) {
      const filePath = url.slice(7);
      const buffer = await fs.promises.readFile(filePath);
      const filename = path.basename(filePath);
      return { buffer, filename };
    }

    if (url.startsWith('base64://')) {
      const buffer = Buffer.from(data.url.slice(9), 'base64');
      let filename = data.file;
      if (!filename) {
        if (!url.startsWith('base64://')) {
          filename = path.basename(url);
        } else {
          const fileType = await FileType.fromBuffer(buffer);
          if (fileType) {
            filename = `media.${fileType.ext}`;
          } else {
            filename = 'media.bin';
          }
        }
      }
      return { buffer, filename };
    }

    const info = await this.bot.http.file(url);
    return { buffer: Buffer.from(info.data), filename: info.filename };
  }

  private text(content: string) {
    this.buffer += content;
  }

  private readonly cardPropertyArrayKeyList = {
    action_menu: 'action_list',
    check_box: 'option_list',
    button_selection: 'option_list',
  };

  private readonly cardPropertyArrayPartentKeyList = {
    select_list: 'option_list',
  };

  private parseCardProperties(element: Element, parentKey = '') {
    const data: any = transformKey(element.attrs, snakeCase);
    const topKey = snakeCase(element.type);
    const arrayKey =
      this.cardPropertyArrayKeyList[topKey] ||
      this.cardPropertyArrayPartentKeyList[parentKey];
    for (const prop of element.children) {
      if (arrayKey && topKey !== arrayKey) {
        data[arrayKey] ??= [];
        data[arrayKey].push(this.parseCardProperties(prop, topKey));
      }
      if (!prop.type) {
        continue;
      }
      const key = snakeCase(prop.type);
      let value: any;
      if (key.endsWith('_list')) {
        value = prop.children.map((item) =>
          this.parseCardProperties(item, key),
        );
      } else {
        value = this.parseCardProperties(prop, topKey);
      }
      data[key] = value;
    }
    if (!data.task_id && element.type.startsWith('wecom:')) {
      data.task_id = cryptoRandomString({ length: 127, type: 'alphanumeric' });
    }
    return data;
  }

  async flush() {
    const content = this.buffer.trim();
    if (content) {
      const elem = segment.text(content);
      if (this.isMarkdown) {
        elem.type = 'markdown';
      }
      await this.post(elem);
      this.buffer = '';
    }
  }

  private async post(element: Element) {
    try {
      const messageId = await this.postElement(element);
      if (!messageId) return;
      const session = this.bot.session();
      session.messageId = messageId;
      session.app.emit(session, 'send', session);
      this.results.push(session);
    } catch (e) {
      this.errors.push(e);
    }
  }

  private async postElement(element: Element) {
    const { type, attrs } = element;
    switch (type) {
      case 'text':
        return this.sendTextMessage(attrs.content);
      case 'markdown':
        return this.sendMarkdownMessage(attrs.content);
      case 'image':
      case 'video':
      case 'file':
      case 'voice':
        if (!attrs.url) {
          return;
        }
        const { buffer, filename } = await this.prepareBufferAndFilename(
          type,
          attrs,
        );
        const mediaId = await this.uploadMedia(buffer, type, filename);
        if (!mediaId) {
          break;
        }
        return this.sendMediaMessage(type, filename, buffer);
      case 'wecom:card':
        return this.sendGenericMessage({
          msgtype: 'template_card',
          template_card: this.parseCardProperties(element),
        });
      default:
        if (type.startsWith('wecom:')) {
          const wecomMessageType = type.slice(6);
          return this.sendGenericMessage({
            msgtype: wecomMessageType,
            [wecomMessageType]: transformKey(attrs, snakeCase),
          });
        }
        return;
    }
  }

  private isMarkdown = false;

  async visit(element: segment) {
    const { type, attrs, children } = element;
    switch (type) {
      case 'text':
        this.text(attrs.content);
        break;
      case 'p':
        await this.render(children);
        this.text('\n');
        break;
      case 'a':
        await this.render(children);
        if (attrs.href) this.text(` (${attrs.href}) `);
        break;
      case 'at':
        if (attrs.id) {
          this.text(`@${attrs.id}`);
        } else if (attrs.type === 'all') {
          this.text('@全体成员');
        } else if (attrs.type === 'here') {
          this.text('@在线成员');
        } else if (attrs.role) {
          this.text(`@${attrs.role}`);
        }
        break;
      case 'sharp':
        this.text(` #${attrs.name} `);
        break;
      case 'message':
        await this.flush();
        const prevIsMarkdown = this.isMarkdown;
        this.isMarkdown =
          prevIsMarkdown || !!(attrs.markdown && attrs.markdown !== 0);
        await this.render(children);
        await this.flush();
        this.isMarkdown = prevIsMarkdown;
        break;
      default:
        if (
          type.startsWith('wecom:') ||
          ['image', 'video', 'file', 'voice'].includes(type)
        ) {
          await this.flush();
          await this.post(element);
        }
        await this.render(children);
    }
  }
}
