import { Adapter, Logger } from 'koishi';
import { DefinePlugin, InjectLogger, KoaContext } from 'koishi-thirdeye';
import type WeComBot from './index';
import { XMLParser } from 'fast-xml-parser';
import {
  WecomEventBody,
  WecomEventResponse,
  WecomReceiveMessageDto,
  WecomRegisterDto,
} from './def';
import { decrypt, getSignature } from '@wecom/crypto';
import rawBody from 'raw-body';
import { dispatchSession } from './utils';

@DefinePlugin()
export class WeComAdapter extends Adapter.Server<WeComBot> {
  private xmlParser = new XMLParser();

  @InjectLogger()
  private logger: Logger;

  private checkSignature(
    token: string,
    dto: WecomReceiveMessageDto,
    body: string,
  ) {
    const { msg_signature, timestamp, nonce } = dto;
    const signature = getSignature(token, timestamp, nonce, body);
    return signature === msg_signature;
  }

  async start(bot: WeComBot) {
    bot.ctx.router.get(bot.config.path, this.koaGetHandler.bind(this));
    bot.ctx.router.post(bot.config.path, this.koaPostHandler.bind(this));
    return bot.initialize();
  }
  async stop(bot: WeComBot) {}

  async koaGetHandler(ctx: KoaContext) {
    const query = ctx.request.query as unknown as WecomRegisterDto;
    const { echostr } = query;
    const bot = this.bots.find((bot) =>
      this.checkSignature(bot.config.token, query, echostr),
    );
    if (!bot) {
      this.logger.warn('Bot not found.');
      ctx.status = 404;
      ctx.body = 'Bot not found.';
      return;
    }
    const decrypted = decrypt(bot.config.encodingAESKey, echostr);
    const message = decrypted?.message;
    if (!message) {
      this.logger.warn('Invalid message: %s', decrypted);
      ctx.status = 400;
      ctx.body = 'invalid message';
      return;
    }
    this.logger.success(`Registered bot ${bot.selfId}: ${message}`);
    ctx.body = message;
  }

  async koaPostHandler(ctx: KoaContext) {
    const query = ctx.request.query as unknown as WecomReceiveMessageDto;
    const rawData = (await rawBody(ctx.req)).toString('utf8').trim();
    const { xml: parsedData } = (await this.xmlParser.parse(rawData)) as {
      xml: WecomEventResponse;
    };
    if (!parsedData?.Encrypt) {
      this.logger.warn('Invalid xml: %s', rawData);
      ctx.status = 400;
      ctx.body = 'invalid message';
      return;
    }

    const bot = this.bots.find(
      (bot) =>
        bot.config.agentId === parsedData.AgentID?.toString() &&
        bot.config.corpId === parsedData.ToUserName,
    );
    if (!bot) {
      ctx.status = 404;
      ctx.body = 'Bot not found.';
      return;
    }

    if (!this.checkSignature(bot.config.token, query, parsedData.Encrypt)) {
      this.logger.warn(`Invalid signature for bot ${bot.selfId}`);
      ctx.status = 403;
      ctx.body = 'invalid signature';
      return;
    }

    parsedData.data = decrypt(bot.config.encodingAESKey, parsedData.Encrypt);
    if (!parsedData.data) {
      this.logger.warn('Invalid decrypted message: %s', parsedData.Encrypt);
      ctx.status = 400;
      ctx.body = 'invalid message';
      return;
    }
    parsedData.body = this.xmlParser.parse(parsedData.data.message)
      .xml as WecomEventBody;
    if (!parsedData.body) {
      this.logger.warn(
        'Invalid decrypted xml message: %s',
        parsedData.data.message,
      );
    }
    dispatchSession(bot, parsedData);
    ctx.body = 'success';
    ctx.status = 200;
  }
}
