import { Adapter, Logger } from 'koishi';
import { AdapterConfig, dispatchSession } from './utils';
import { BotConfig, WeComBot } from './bot';
import { decrypt, getSignature } from '@wecom/crypto';
import {
  WecomEventResponse,
  WecomReceiveMessageDto,
  WecomRegisterDto,
  WecomResponse,
} from './def';
import rawBody from 'raw-body';
import { XMLParser } from 'fast-xml-parser';

const logger = new Logger('wecom');

export default class HttpServer extends Adapter<BotConfig, AdapterConfig> {
  private xmlParser = new XMLParser();
  static schema = BotConfig;

  stop() {
    logger.debug('http server closing');
  }

  private checkSignature(dto: WecomReceiveMessageDto, body: string) {
    const { msg_signature, timestamp, nonce } = dto;
    const signature = getSignature(this.config.token, timestamp, nonce, body);
    return {
      valid: signature === msg_signature,
      signature,
      msg_signature,
    };
  }

  public bots: WeComBot[];

  async connect(bot: WeComBot) {
    return bot.initialize();
  }

  async start() {
    const { path } = this.config;
    this.ctx.router.get(path, async (ctx) => {
      const query = ctx.request.query as unknown as WecomRegisterDto;
      const { echostr } = query;
      const signatureResult = this.checkSignature(query, echostr);
      if (!signatureResult.valid) {
        logger.warn(
          `Invalid signature: ${signatureResult.msg_signature} vs ${signatureResult.signature}`,
        );
        ctx.status = 403;
        ctx.body = 'invalid signature';
        return;
      }
      const decrypted = decrypt(this.config.encodingAESKey, echostr);
      const message = decrypted?.message;
      if (!message) {
        logger.warn('Invalid message: %s', decrypted);
        ctx.status = 400;
        ctx.body = 'invalid message';
        return;
      }
      logger.success(`Registered: ${message}`);
      ctx.body = message;
    });

    this.ctx.router.post(path, async (ctx) => {
      const query = ctx.request.query as unknown as WecomReceiveMessageDto;
      const rawData = (await rawBody(ctx.req)).toString('utf8').trim();
      const { xml: parsedData } = (await this.xmlParser.parse(rawData)) as {
        xml: WecomResponse<any>;
      };
      if (!parsedData?.Encrypt) {
        logger.warn('Invalid xml: %s', rawData);
        ctx.status = 400;
        ctx.body = 'invalid message';
        return;
      }

      const signatureResult = this.checkSignature(query, parsedData.Encrypt);
      if (!signatureResult.valid) {
        logger.warn(
          `Invalid signature: ${signatureResult.msg_signature} vs ${signatureResult.signature}`,
        );
        ctx.status = 403;
        ctx.body = 'invalid signature';
        return;
      }
      const bot = this.bots.find(
        (bot) =>
          bot.config.agentId === parsedData.AgentID?.toString() &&
          bot.config.corpId === parsedData.ToUserName,
      );
      if (!bot) {
        ctx.status = 404;
        ctx.body = 'Bot not found.';
        return;
      }
      parsedData.data = decrypt(this.config.encodingAESKey, parsedData.Encrypt);
      if (!parsedData.data) {
        logger.warn('Invalid decrypted message: %s', parsedData.Encrypt);
        ctx.status = 400;
        ctx.body = 'invalid message';
        return;
      }
      parsedData.body = this.xmlParser.parse(parsedData.data.message)
        .xml as WecomEventResponse;
      if (!parsedData.body) {
        logger.warn(
          'Invalid decrypted xml message: %s',
          parsedData.data.message,
        );
      }
      dispatchSession(bot, parsedData);
      ctx.body = 'success';
      ctx.status = 200;
    });
  }
}
