import { createHash } from 'crypto';
import { Schema } from 'koishi';

export type WecomButtonTypes =
  | 'click'
  | 'view'
  | 'scancode_push'
  | 'scancode_waitmsg'
  | 'pic_sysphoto'
  | 'pic_photo_or_album'
  | 'pic_weixin'
  | 'location_select'
  | 'view_miniprogram';

export interface WecomButton {
  type: WecomButtonTypes;
  name: string;
  key?: string;
  url?: string;
  pagepath?: string;
  appid?: string;
}

export interface WecomParentButton {
  name: string;
  sub_button: WecomButton[];
}

export interface WecomMenuDef {
  type: WecomButtonTypes | 'parent';
  name: string;
  command?: string;
  children?: (Omit<WecomMenuDef, 'children'> & { type: WecomButtonTypes })[];
  url?: string;
  pagePath?: string;
  mediaId?: string;
}

// eslint-disable-next-line @typescript-eslint/ban-ts-comment
// @ts-ignore
export const WecomMenuDef: Schema<WecomMenuDef> = Schema.object({
  type: Schema.union([
    'parent',
    'click',
    'view',
    'scancode_push',
    'scancode_waitmsg',
    'pic_sysphoto',
    'pic_photo_or_album',
    'pic_weixin',
    'location_select',
    'view_miniprogram',
  ])
    .required()
    .description('菜单的响应动作类型。'),
  name: Schema.string().description('菜单标题。').required(),
  command: Schema.string().description('菜单的响应动作指令。'),
  children: Schema.array(
    Schema.object({
      type: Schema.union([
        'click',
        'view',
        'scancode_push',
        'scancode_waitmsg',
        'pic_sysphoto',
        'pic_photo_or_album',
        'pic_weixin',
        'location_select',
        'view_miniprogram',
      ])
        .required()
        .description('菜单的响应动作类型。'),
      name: Schema.string().description('菜单标题。').required(),
      command: Schema.string().description('菜单的响应动作指令。'),
      url: Schema.string().description(
        '网页链接，成员点击菜单可打开链接，不超过1024字节。',
      ),
      pagePath: Schema.string().description('小程序的页面路径。'),
      appId: Schema.string().description('小程序的appid。'),
    }),
  ).description('子菜单。'),
  url: Schema.string().description(
    '网页链接，成员点击菜单可打开链接，不超过1024字节。',
  ),
  pagePath: Schema.string().description('小程序的页面路径。'),
  appId: Schema.string().description('小程序的appid。'),
});

export const adaptMenu = (
  menu: WecomMenuDef,
  registerMap?: Map<string, string>,
): WecomButton | WecomParentButton => {
  if (!menu.type && !menu.children) {
    throw new Error('Menu type or children must be set');
  }
  if (menu.type !== 'parent') {
    const key = createHash('sha512')
      .update(`${menu.type}${menu.name}${menu.command || ''}`)
      .digest('hex');
    if (menu.command) {
      registerMap?.set(key, menu.command);
    }
    return {
      type: menu.type as WecomButtonTypes,
      name: menu.name,
      key,
      url: menu.url,
      pagepath: menu.pagePath,
      appid: menu.mediaId,
    };
  } else {
    return {
      name: menu.name,
      sub_button: menu.children.map((c) => adaptMenu(c, registerMap)),
    } as WecomParentButton;
  }
};

export interface WecomMenu {
  button: WecomButton[];
}
