import { Bot, Schema, segment, Session } from 'koishi';
import { WeComBot } from './bot';
import {
  WecomChatBody,
  WecomEventResponse,
  WecomLinkBody,
  WecomLocationMessageBody,
  WecomPicBody,
  WeComUser,
  WecomVideoBody,
  WecomVoiceBody,
} from './def';

export interface AdapterConfig {
  path?: string;
  token?: string;
  encodingAESKey?: string;
}

export const AdapterConfig: Schema<AdapterConfig> = Schema.object({
  path: Schema.string().description('企业微信回调路径。').default('/wecom'),
  token: Schema.string()
    .role('secret')
    .required()
    .description('应用消息上报 token。'),
  encodingAESKey: Schema.string()
    .role('secret')
    .required()
    .description('应用消息上播 AES 密钥。'),
});

export function adaptUser(user: WeComUser): Bot.User {
  return {
    userId: user.userid,
    username: user.name,
    nickname: user.name,
    avatar: user.avatar,
  };
}

export function adaptSession(bot: WeComBot, input: WecomEventResponse) {
  const { body } = input;
  if (!body) {
    return;
  }
  const session: Partial<Session> = {
    selfId: bot.selfId,
    targetId: body.ToUserName,
    messageId: body.MsgId?.toString(),
    wecom: body,
    userId: body.FromUserName,
    channelId: body.FromUserName,
    timestamp: body.CreateTime,
  };
  if (body.MsgType === 'event') {
    session.type = `wecom/${body.Event}`;
  } else {
    switch (body.MsgType) {
      case 'text':
        const textBody = body as WecomChatBody;
        session.content = textBody.Content;
        break;
      case 'image':
        const imageBody = body as WecomPicBody;
        session.content = segment('image', {
          url: imageBody.PicUrl,
        });
        break;
      case 'link':
        const linkBody = body as WecomLinkBody;
        session.content = linkBody.Url;
        break;
      case 'video':
        const videoBody = body as WecomVideoBody;
        session.content = segment('wecom:video', {
          mediaId: videoBody.MediaId,
          thumbMediaId: videoBody.ThumbMediaId,
        });
        break;
      case 'voice':
        const voiceBody = body as WecomVoiceBody;
        session.content = segment('voice', {
          mediaId: voiceBody.MediaId,
          format: voiceBody.Format,
        });
        break;
      case 'location':
        const locationBody = body as WecomLocationMessageBody;
        session.content = segment('location', {
          latitude: locationBody.Location_X,
          longitude: locationBody.Location_Y,
          scale: locationBody.Scale,
          label: locationBody.Label,
        });
        break;
      default:
        return;
    }
    session.type = 'message';
    session.subtype = 'private';
    session.author = {
      userId: session.userId,
    };
  }
  return session;
}

export function dispatchSession(bot: WeComBot, message: WecomEventResponse) {
  const payload = adaptSession(bot, message);
  if (!payload) return;
  const session = new Session(bot, payload);
  session.wecom = message.body;
  bot.adapter.dispatch(session);
}
