import { createHash } from 'crypto';
import { Schema } from 'koishi';

export const WechatButtonTypes = [
  'click',
  'view',
  'miniprogram',
  'scancode_push',
  'scancode_waitmsg',
  'pic_sysphoto',
  'pic_photo_or_album',
  'pic_weixin',
  'location_select',
  'media_id',
  'view_limited',
  'article_id',
  'article_view_limited',
] as const;

export type WechatButtonTypes = typeof WechatButtonTypes[number];

export interface WechatButton {
  type: WechatButtonTypes;
  name: string;
  key?: string;
  url?: string;
  pagepath?: string;
  appid?: string;
}

export interface WechatParentButton {
  name: string;
  sub_button: WechatButton[];
}

export interface WechatMenuDef {
  type: WechatButtonTypes | 'parent';
  name: string;
  command?: string;
  children?: (Omit<WechatMenuDef, 'children'> & { type: WechatButtonTypes })[];
  url?: string;
  pagePath?: string;
  mediaId?: string;
}

const menuInfoSchemas = {
  name: Schema.string().description('菜单标题。').required(),
  command: Schema.string().description('菜单的响应动作指令。'),
  url: Schema.string().description(
    '网页链接，成员点击菜单可打开链接，不超过1024字节。',
  ),
  pagePath: Schema.string().description('小程序的页面路径。'),
  appId: Schema.string().description('小程序的appid。'),
};

// eslint-disable-next-line @typescript-eslint/ban-ts-comment
// @ts-ignore
export const WechatMenuDef: Schema<WechatMenuDef> = Schema.object({
  type: Schema.union(['parent', ...WechatButtonTypes])
    .required()
    .description('菜单的响应动作类型。'),
  ...menuInfoSchemas,
  children: Schema.array(
    Schema.object({
      type: Schema.union(WechatButtonTypes)
        .required()
        .description('菜单的响应动作类型。'),
      ...menuInfoSchemas,
    }),
  ).description('子菜单。'),
});

export const adaptMenu = (
  menu: WechatMenuDef,
  registerMap?: Map<string, string>,
): WechatButton | WechatParentButton => {
  if (!menu.type && !menu.children) {
    throw new Error('Menu type or children must be set');
  }
  if (menu.type !== 'parent') {
    const key = createHash('sha512')
      .update(`${menu.type}${menu.name}${menu.command || ''}`)
      .digest('hex');
    if (menu.command) {
      registerMap?.set(key, menu.command);
    }
    return {
      type: menu.type as WechatButtonTypes,
      name: menu.name,
      key,
      url: menu.url,
      pagepath: menu.pagePath,
      appid: menu.mediaId,
    };
  } else {
    return {
      name: menu.name,
      sub_button: menu.children.map((c) => adaptMenu(c, registerMap)),
    } as WechatParentButton;
  }
};

export interface WechatMenu {
  button: WechatButton[];
}
