import { Logger, Messenger, Random, segment } from 'koishi';
import WechatBot from './index';

interface ConsumerMessagePayload {
  msgtype: 'text' | 'image' | 'voice' | 'video';
  touser?: string;
  text?: {
    content: string;
  };
  image?: {
    media_id: string;
  };
  voice?: {
    media_id: string;
  };
  video?: {
    media_id: string;
    thumb_media_id?: string;
    title: string;
    description: string;
  };
}

export class WechatConsumerMessenger extends Messenger<WechatBot> {
  logger = new Logger('wechat-consumer-messenger');
  async sendConsumer(payload: ConsumerMessagePayload) {
    const token = await this.bot.internal.getAccessToken();
    let success = false;
    try {
      const result = await this.bot.http.post<{
        errcode: number;
        errmsg: string;
      }>(
        'https://api.weixin.qq.com/cgi-bin/message/custom/send',
        {
          touser: this.channelId,
          ...payload,
        },
        {
          params: {
            access_token: token,
          },
        },
      );
      if (result.errcode > 0) {
        this.logger.warn(
          `Send consumer message failed: ${result.errcode} ${result.errmsg}`,
        );
      } else {
        success = true;
      }
    } catch (e) {
      this.logger.error(`Send consumer message errored: ${e.message}`);
    }
    if (success) {
      const session = this.bot.session();
      session.messageId = Random.id();
      session.app.emit(session, 'send', session);
      this.results.push(session);
    }
  }

  private buffer = '';

  async flush() {
    const content = this.buffer.trim();
    if (content) {
      await this.sendConsumer({
        msgtype: 'text',
        text: {
          content,
        },
      });
    }
  }

  text(text: string) {
    this.buffer += text;
  }

  async visit(element: segment) {
    const { type, attrs, children } = element;
    switch (type) {
      case 'text':
        this.text(attrs.content);
        break;
      case 'p':
        await this.render(children);
        this.text('\n');
        break;
      case 'a':
        this.text(element.toString());
        break;
      case 'at':
        if (attrs.id) {
          this.text(`@${attrs.id}`);
        } else if (attrs.type === 'all') {
          this.text('@全体成员');
        } else if (attrs.type === 'here') {
          this.text('@在线成员');
        } else if (attrs.role) {
          this.text(`@${attrs.role}`);
        }
        break;
      case 'sharp':
        this.text(` #${attrs.name} `);
        break;
      case 'message':
        await this.flush();
        await this.render(children);
        await this.flush();
        break;
      case 'image':
      case 'voice':
      case 'video':
        await this.flush();
        const mediaId = await this.bot.uploadMedia(element);
        if (mediaId) {
          await this.sendConsumer({
            msgtype: type,
            [type]: {
              media_id: mediaId,
              title: attrs.title,
              description: attrs.description,
            },
          });
        }
        break;
      default:
        await this.render(children);
    }
  }
}
