// import 'source-map-support/register';
import {
  DefinePlugin,
  Inject,
  InjectLogger,
  PluginDef,
  PluginSchema,
  ReplySession,
  Reusable,
  UsePlugin,
} from 'koishi-thirdeye';
import { WechatOfficialConfig } from './config';
import {
  Bot,
  Context,
  Element,
  Logger,
  Quester,
  segment,
  SendOptions,
  Session,
  Universal,
} from 'koishi';
import WechatApplication from 'koa-wechat-public';
import { adaptMenu } from './utils/menu';
import { Send } from 'koa-wechat-public/typings/send';
import { getFirstElement, getPlainText } from './utils/message';
import FormData from 'form-data';
import * as fs from 'fs';
import ext2mime from 'ext2mime';
import mime2ext from 'mime2ext';
import FileType from 'file-type';
import path from 'path';
import { Readable } from 'stream';
import { WechatAdapter } from './adapter';
import XmlParser from 'koa-xml-body';
import { WechatConsumerMessenger } from './message';
export * from './config';

declare module 'koishi' {
  interface Session {
    wechatCtx: WechatApplication.ApplicationCommonContext &
      WechatApplication.ApplicationEventContext;
  }
  interface Events {
    'wechat:subscribe'(session: ReplySession): void;
    'wechat:unsubscribe'(session: ReplySession): void;
  }
}

@Reusable()
@PluginSchema(WechatOfficialConfig)
@DefinePlugin({ name: 'adapter-wechat-official' })
export default class WechatBot extends Bot<Partial<WechatOfficialConfig>> {
  internal: WechatApplication;
  private menuMap = new Map<string, string>();
  @Inject()
  http: Quester;
  @InjectLogger()
  private logger: Logger;

  constructor(ctx: Context, config: Partial<WechatOfficialConfig>) {
    super(ctx, config);
  }

  @UsePlugin()
  private loadAdapter() {
    this.internal = new WechatApplication(
      this.config as WechatApplication.WechatApplicationConfig,
    );
    return PluginDef(WechatAdapter, this);
  }

  async initialize() {
    this.ctx.router.get(this.config.path, this.internal.auth());
    const xmlParser = XmlParser();
    const handler = this.internal.handle();
    this.ctx.router.post(this.config.path, async (ctx, next) => {
      ctx.request.body = undefined;
      // eslint-disable-next-line @typescript-eslint/no-empty-function
      await xmlParser(ctx, async () => {});
      await handler(ctx, next);
      if (!ctx.response.body) {
        ctx.response.body = 'success';
      }
    });
    if (this.config.menus?.length) {
      this.internal.createMenu({
        button: this.config.menus.map((menu) => adaptMenu(menu, this.menuMap)),
      });
      await this.internal['menuHandler']();
    }
    this.online();
  }

  private async putFile(form: FormData, url: string) {
    if (url.startsWith('file://')) {
      const filepath = url.slice(7);
      const size = (await fs.promises.stat(filepath)).size;
      form.append('media', fs.createReadStream(filepath), {
        knownLength: size,
        filepath,
        contentType: ext2mime(path.extname(filepath)),
      });
    } else if (url.startsWith('base64://')) {
      const base64 = url.slice(9);
      const buf = Buffer.from(base64, 'base64');
      const fileType = await FileType.fromBuffer(buf);
      form.append('media', buf, {
        knownLength: buf.length,
        filename: `file.${fileType?.ext || 'bin'}`,
        contentType: fileType?.mime || 'application/octet-stream',
      });
    } else if (url.startsWith('data:')) {
      const [, type, base64] = url.match(/^data:(.*?);base64,(.*)$/);
      const buf = Buffer.from(base64, 'base64');
      form.append('media', buf, {
        knownLength: buf.length,
        contentType: type,
        filename: `file.${mime2ext(type) || 'bin'}`,
      });
    } else {
      const resp = await this.http.axios<Readable>(url, {
        responseType: 'stream',
      });
      form.append('media', resp.data, {
        knownLength: Number(resp.headers['content-length']),
        contentType: resp.headers['content-type'] || 'application/octet-stream',
        filename: `file.${mime2ext(resp.headers['content-type']) || 'bin'}`,
      });
    }
  }

  async uploadMedia(element: Element) {
    try {
      const { type } = element;
      const uploadType = type === 'audio' ? 'voice' : type;
      const form = new FormData();
      await this.putFile(form, element.attrs.url);
      const headers = form.getHeaders();
      headers['content-length'] = await new Promise((resolve, reject) => {
        form.getLength((err, length) => {
          if (err) return reject(err);
          resolve(length);
        });
      });
      const resp = await this.http.post<{
        type: string;
        media_id: string;
        created_at: number;
        errcode: number;
        errmsg: string;
      }>('https://api.weixin.qq.com/cgi-bin/media/upload', form, {
        responseType: 'json',
        params: {
          access_token: await this.internal.getAccessToken(),
          type: uploadType,
        },
        headers,
      });
      if (resp.media_id) {
        return resp.media_id;
      }
      this.logger.warn(
        `Failed to upload media ${element.attrs.url}: ${resp.errcode} ${resp.errmsg}`,
      );
      return;
    } catch (e) {
      this.logger.error(
        `Failed to upload media ${element.attrs.url}: ${e.message}`,
      );
    }
  }

  private async setReply(sender: Send, reply: Element[]) {
    const firstElement = getFirstElement(reply);
    if (firstElement.type === 'image') {
      const mediaId = await this.uploadMedia(firstElement);
      if (mediaId) {
        sender.sendImageMsg(mediaId);
        return;
      }
    } else if (firstElement.type === 'video') {
      const mediaId = await this.uploadMedia(firstElement);
      if (mediaId) {
        sender.sendVideoMsg(
          mediaId,
          firstElement.attrs.title,
          firstElement.attrs.desc,
        );
        return;
      }
    }
    const plainText = getPlainText(reply);
    if (plainText) {
      sender.sendTxtMsg(plainText);
    }
  }

  async handleSession(
    acc:
      | WechatApplication.ApplicationCommonContext
      | WechatApplication.ApplicationEventContext,
    payload: Partial<Session>,
  ) {
    const session = new ReplySession(this, payload);
    session.wechatCtx = acc as any;
    session.timestamp = acc.createTime;
    session.targetId = acc.toUser;
    session.userId = acc.fromUser;
    session.messageId = acc['msgId'] || `${acc.fromUser}:${acc.createTime}`;
    session.channelId = acc.fromUser;
    if (session.type === 'message') {
      session.subtype = 'private';
      session.author = { userId: acc.fromUser };
    }
    const reply = await session.process(4500);
    if (reply?.length) {
      await this.setReply(acc.send, segment.normalize(reply));
    }
  }

  async handleMenu(acc: WechatApplication.ApplicationEventContext) {
    const menuContent = this.menuMap.get(acc.eventKey);
    if (!menuContent) return;
    return this.handleSession(acc, {
      type: 'message',
      content: menuContent,
    });
  }

  async transformVideo(mediaId: string) {
    try {
      const data = await this.http.get<{
        video_url: string;
        errcode: number;
        errmsg: string;
      }>('https://api.weixin.qq.com/cgi-bin/media/get', {
        responseType: 'json',
        params: {
          access_token: await this.internal.getAccessToken(),
          media_id: mediaId,
        },
      });
      if (!data.video_url) {
        this.logger.warn(
          `Failed to transform video ${mediaId}: ${data.errcode} ${data.errmsg}`,
        );
      }
      return segment.video(data.video_url);
    } catch (e) {
      this.logger.error(`Failed to transform video ${mediaId}: ${e.message}`);
      return '';
    }
  }

  async transformVoice(mediaId: string) {
    try {
      const data = await this.http.axios<ArrayBuffer>(
        'https://api.weixin.qq.com/cgi-bin/media/get',
        {
          method: 'GET',
          responseType: 'arraybuffer',
          params: {
            access_token: await this.internal.getAccessToken(),
            media_id: mediaId,
          },
        },
      );
      return segment.audio(data.data, data.headers['content-type']);
    } catch (e) {
      this.logger.error(`Failed to transform voice ${mediaId}: ${e.message}`);
      return '';
    }
  }

  // message.ts
  async sendMessage(
    channelId: string,
    content: segment.Fragment,
    guildId?: string,
    options?: SendOptions,
  ): Promise<string[]> {
    return this.sendPrivateMessage(channelId, content, options);
  }
  async sendPrivateMessage(
    userId: string,
    content: segment.Fragment,
    options?: SendOptions,
  ): Promise<string[]> {
    return new WechatConsumerMessenger(this, userId, undefined, options).send(
      content,
    );
  }
  async getMessage(
    channelId: string,
    messageId: string,
  ): Promise<Universal.Message> {
    return;
  }
  async getMessageList(
    channelId: string,
    before?: string,
  ): Promise<Universal.Message[]> {
    return [];
  }
  async editMessage(
    channelId: string,
    messageId: string,
    content: segment.Fragment,
  ): Promise<void> {
    return;
  }
  async deleteMessage(channelId: string, messageId: string): Promise<void> {
    return;
  }

  // user
  async getSelf(): Promise<Universal.User> {
    return { userId: this.selfId };
  }
  async getUser(userId: string): Promise<Universal.User> {
    return;
  }
  async getFriendList(): Promise<Universal.User[]> {
    return [];
  }
  async deleteFriend(userId: string): Promise<void> {
    return;
  }

  // guild
  async getGuild(guildId: string): Promise<Universal.Guild> {
    return;
  }
  async getGuildList(): Promise<Universal.Guild[]> {
    return [];
  }

  // guild member
  async getGuildMember(
    guildId: string,
    userId: string,
  ): Promise<Universal.GuildMember> {
    return;
  }
  async getGuildMemberList(guildId: string): Promise<Universal.GuildMember[]> {
    return [];
  }
  async kickGuildMember(
    guildId: string,
    userId: string,
    permanent?: boolean,
  ): Promise<void> {
    return;
  }
  async muteGuildMember(
    guildId: string,
    userId: string,
    duration: number,
    reason?: string,
  ): Promise<void> {
    return;
  }

  // channel
  async getChannel(
    channelId: string,
    guildId?: string,
  ): Promise<Universal.Channel> {
    return;
  }
  async getChannelList(guildId: string): Promise<Universal.Channel[]> {
    return [];
  }
  async muteChannel(
    channelId: string,
    guildId?: string,
    enable?: boolean,
  ): Promise<void> {
    return;
  }

  // request
  async handleFriendRequest(
    messageId: string,
    approve: boolean,
    comment?: string,
  ): Promise<void> {
    return;
  }
  async handleGuildRequest(
    messageId: string,
    approve: boolean,
    comment?: string,
  ): Promise<void> {
    return;
  }
  async handleGuildMemberRequest(
    messageId: string,
    approve: boolean,
    comment?: string,
  ): Promise<void> {
    return;
  }
}
