import 'source-map-support/register';
import { Context, Schema, Session } from 'koishi';
import { Show } from './playbook/Show';
import { Playbook } from './playbook/Playbook';
import { plainToClass } from 'class-transformer';
import loadJsonFile from 'load-json-file';
import { MaybeArray } from 'koishi';
import type { OneBotBot } from '@koishijs/plugin-adapter-onebot';

const selectors = [
  'user',
  'guild',
  'channel',
  'self',
  'private',
  'platform',
] as const;

type SelectorType = typeof selectors[number];
type SelectorValue = boolean | MaybeArray<string | number>;
type BaseSelection = { [K in SelectorType as `$${K}`]?: SelectorValue };

export interface Selection extends BaseSelection {
  $and?: Selection[];
  $or?: Selection[];
  $not?: Selection;
}

export interface PluginConfig {
  adminContext?: Selection;
  autoChangeName?: boolean;
  dropHelp?: boolean;
  playbookPathPrefix?: string;
}

export class MyPlugin {
  config: PluginConfig;
  ctx: Context;
  adminCtx: Context;
  shows = new Map<string, Show>();

  name = 'act-main';
  schema: Schema<PluginConfig> = Schema.object({
    adminContext: Schema.any().description('管理员接口作用域。'),
    autoChangeName: Schema.boolean()
      .description('公演开始之前是否修改每个演员机器人的群名片。推荐开启。')
      .default(true),
    playbookPathPrefix: Schema.string()
      .description(
        '公演剧本存放路径。公演剧本的文件名是 `/path/to/playbookPathPrefix/name.json` 对应于 name 剧本。',
      )
      .default('./playbooks'),
    dropHelp: Schema.boolean()
      .description('是否删除 `help` 命令，避免社死 (?)')
      .default(false),
  });
  apply(ctx: Context, config: PluginConfig) {
    this.ctx = ctx;
    this.config = config;
    if (this.config.dropHelp) {
      ctx.command('help').dispose();
    }
    this.adminCtx = this.ctx.select(this.config.adminContext);
    const showComamnd = this.adminCtx
      .command('act [groupId:string]', '获取公演状态')
      .usage('不带参数获取所有正在公演的群，带参数则获取特定群。')
      .action((argv, groupId) => {
        if (!groupId) {
          return `当前正在公演的群有:\n${Array.from(this.shows.keys()).join(
            ' ',
          )}`;
        }
        if (!this.shows.has(groupId)) {
          return `群 ${groupId} 并不在公演哦。`;
        }
        const show = this.shows.get(groupId);
        return show.getStatus();
      });

    showComamnd
      .subcommand(
        '.create <groupId:string> <playbookFilename:string> [...specificCharacters]',
        '创建公演',
      )
      .usage(
        'groupId 目标群。playbookFilename 是公演剧本文件，需要在服务器放好。后面的参数可以以 人物名=帐号 的形式指定特定的人物，否则随机分配。',
      )
      .example(
        'act.create 11111111 play1 幽幽子=2222222 在群 11111111 创建一个公演，剧本是 play1，同时指定人物幽幽子为 2222222 扮演。',
      )
      .action(
        async (argv, groupId, playbookFilename, ...specificCharacters) => {
          if (!groupId || !playbookFilename) {
            return '缺少参数。';
          }
          return this.createShow(
            argv.session,
            groupId,
            playbookFilename,
            specificCharacters,
          );
        },
      );
    showComamnd
      .subcommand('.delete <groupId:string>', '停止')
      .usage('groupId 目标群。')
      .example('act.delete 11111111 停止群 11111111 的公演。')
      .action((argv, groupId) => {
        if (!groupId) {
          return '缺少参数。';
        }
        if (!this.shows.has(groupId)) {
          return `群 ${groupId} 并不在公演哦。`;
        }
        const show = this.shows.get(groupId);
        show.endShow(`Ended by user ${argv.session.userId}`);
        return '停止成功。';
      });
  }

  async createShow(
    session: Session,
    groupId: string,
    playbookFilename: string,
    specificCharacters: string[] = [],
  ) {
    if (this.shows.has(groupId)) {
      return `群 ${groupId} 正在演呢！`;
    }
    const playbookPath = `${this.config.playbookPathPrefix}/${playbookFilename}.json`;
    let playbook: Playbook;
    try {
      playbook = plainToClass(
        Playbook,
        (await loadJsonFile(playbookPath)) as any,
      );
    } catch (e) {
      return `无法加载剧本文件 ${playbookPath}: ${e.toString()}`;
    }
    const bots: OneBotBot[] = this.ctx.bots.filter(
      (b) => b.adapter.platform === 'onebot',
    ) as OneBotBot[];
    const show = new Show(groupId, playbook, this.config.autoChangeName);
    for (const specificCharacter of specificCharacters) {
      const [characterName, botId] = specificCharacter.split('=');
      if (!botId) {
        return `非法指定人物格式: ${specificCharacter}`;
      }
      const bot = bots.find((b) => b.selfId.toString() === botId);
      if (!bot) {
        return `没有找到机器人 ${botId}`;
      }
      const character = playbook.characters.find(
        (c) => c.id === parseInt(characterName) || c.name === characterName,
      );
      if (!character) {
        return `没有找到人物 ${characterName}`;
      }
      show.useCharacter(character.id, bot);
    }
    if (!(await show.autoCharacters(bots))) {
      return '有人物缺位。';
    }
    show.onFinish = async (message, _show) => {
      this.shows.delete(_show.groupId);
      await session.send(`群 ${_show.groupId} 的公演结束了: ${message}`);
    };
    const launchErrorMessage = await show.launchShow();
    if (launchErrorMessage) {
      return `创建公演失败: ${launchErrorMessage}`;
    }
    this.shows.set(groupId, show);
    return `成功创建群 ${groupId} 的公演 ${playbookFilename}\n出演名单:\n${show.getCharacterList()}`;
  }
}
