import { PlaybookLine } from './PlaybookLine';
import { matchHeader, TuneTimeOptions } from './utility';
import { Character } from './Character';
import { Type } from 'class-transformer';
import 'reflect-metadata';

export class Playbook {
  @Type(() => Character)
  characters: Character[] = [];
  @Type(() => PlaybookLine)
  lines: PlaybookLine[] = [];
  constructor(/*public characterNameMap: Record<string, string> = {}*/) {}
  /*getCharacterDisplayName(name: string) {
    if (this.characterNameMap[name]) {
      return this.characterNameMap[name];
    } else {
      return name;
    }
  }*/
  resetTiming() {
    const firstLineTime = this.lines[0].time;
    if (!firstLineTime) {
      return;
    }
    for (const line of this.lines) {
      line.time -= firstLineTime;
    }
  }
  fixupCharacters() {
    const characterMap = new Map<string, boolean>();
    let currentCharacterIndex = 0;
    for (const line of this.lines) {
      if (!characterMap.has(line.name)) {
        this.characters.push(new Character(currentCharacterIndex++, line.name));
        characterMap.set(line.name, true);
      }
    }
    for (const line of this.lines) {
      line.transferName(this.characters);
    }
  }
  loadText(text: string) {
    const lines = text.trim().split(/\r?\n/);
    let lastLineIndex = -1;
    for (let i = 0; i < lines.length; ++i) {
      const line = lines[i];
      if (!line.length) {
        continue;
      }
      if (matchHeader(line)) {
        if (lastLineIndex >= 0 && i - lastLineIndex > 1) {
          this.lines.push(
            PlaybookLine.FromLines(lines.slice(lastLineIndex, i)),
          );
        }
        lastLineIndex = i;
      }
    }
    const finalIndex = lines.length;
    if (lastLineIndex > 0 && finalIndex - lastLineIndex > 1) {
      this.lines.push(
        PlaybookLine.FromLines(lines.slice(lastLineIndex, finalIndex)),
      );
    }
    this.resetTiming();
    this.fixupCharacters();
  }
  private findPreviousSelfLine(i: number) {
    const currentLine = this.lines[i];
    for (let j = i - 1; j >= 0; j--) {
      const tryLine = this.lines[j];
      if (tryLine.characterId === currentLine.characterId) {
        return tryLine;
      }
    }
    return null;
  }
  tuneTime(options: TuneTimeOptions) {
    if (!this.lines.length) {
      return;
    }
    this.lines[0].time = 0;
    for (let i = 1; i < this.lines.length; ++i) {
      const currentLine = this.lines[i];
      const previousLine = this.lines[i - 1];
      const selfPreviousLine = this.findPreviousSelfLine(i);
      const randomAddValue = Math.random() * options.maxRandomDelay;
      const possibleReadTiming =
        previousLine.time +
        options.readSlope * previousLine.renderContent(this.characters).length +
        options.readIntercept;
      let possibleWriteTiming = 0;
      if (selfPreviousLine) {
        possibleWriteTiming =
          selfPreviousLine.time +
          options.writeSlope *
            currentLine.renderContent(this.characters).length +
          options.writeIntercept;
      }
      currentLine.time = Math.ceil(
        Math.max(possibleReadTiming, possibleWriteTiming) + randomAddValue,
      );
    }
  }
  async play(tick: number) {
    const currentLines = this.lines.filter((l) => l.time === tick);
    for (const line of currentLines) {
      await line.play(this.characters);
    }
    return !this.lines.length || tick >= this.lines[this.lines.length - 1].time;
  }
}
