import { Playbook } from './Playbook';
import type { OneBotBot } from '@koishijs/plugin-onebot/lib/bot';
import moment, { Moment } from 'moment';

export enum ShowStatus {
  Idle,
  Running,
  Finished,
}

export class Show {
  status: ShowStatus = ShowStatus.Idle;
  private characterBotMap = new Map<number, OneBotBot>();
  // eslint-disable-next-line @typescript-eslint/no-empty-function
  onFinish: (message: string, show: Show) => void = () => {};
  private step = 0;
  private launchTime: Moment;
  constructor(
    public groupId: number,
    private playbook: Playbook,
    private autoChangeName = false,
  ) {
    for (const character of playbook.characters) {
      character.playFun = async (line, text) => {
        const bot = this.characterBotMap.get(character.id);
        await bot.internal.sendGroupMsg(this.groupId, text, true);
      };
    }
  }
  private getPerformingBots() {
    return Array.from(this.characterBotMap.values());
  }
  useCharacter(id: number, bot: OneBotBot) {
    this.characterBotMap.set(id, bot);
  }
  isCharacterFull() {
    return this.playbook.characters.every((c) =>
      this.characterBotMap.has(c.id),
    );
  }
  async autoCharacters(bots: OneBotBot[]) {
    let availableBots: OneBotBot[] = [];
    const botNameMap = new Map<string, string>();
    for (const bot of bots) {
      if (bot.adapter.platform !== 'onebot') {
        continue;
      }
      const groups = await bot.internal.getGroupList();
      const matchGroup = groups.find((g) => g.group_id === this.groupId);
      if (matchGroup) {
        availableBots.push(bot);
        botNameMap.set(
          bot.selfId.toString(),
          (await bot.internal.getGroupMemberInfo(this.groupId, bot.selfId))
            .nickname,
        );
      }
    }
    availableBots = availableBots.filter(
      (b) => !this.getPerformingBots().includes(b),
    );
    for (const character of this.playbook.characters) {
      if (this.characterBotMap.has(character.id)) {
        continue;
      }
      // exact
      let matchBot = availableBots.find(
        (b) => botNameMap.get(b.selfId.toString()) === character.name,
      );
      // match 1
      if (!matchBot) {
        matchBot = availableBots.find((b) =>
          botNameMap.get(b.selfId.toString()).includes(character.name),
        );
      }
      // match 2
      if (!matchBot) {
        matchBot = availableBots.find((b) =>
          character.name.includes(botNameMap.get(b.selfId.toString())),
        );
      }
      if (matchBot) {
        this.useCharacter(character.id, matchBot);
        availableBots = availableBots.filter(
          (b) => !this.getPerformingBots().includes(b),
        );
      }
    }
    for (const character of this.playbook.characters) {
      if (this.characterBotMap.has(character.id)) {
        continue;
      }
      const matchBot = availableBots.find((b) => true);
      if (matchBot) {
        this.useCharacter(character.id, matchBot);
        availableBots = availableBots.filter(
          (b) => !this.getPerformingBots().includes(b),
        );
      }
    }
    return this.isCharacterFull();
  }
  private showInterval: NodeJS.Timer;
  async launchShow() {
    if (this.status !== ShowStatus.Idle || !this.isCharacterFull()) {
      return 'Characters missing or not idle.';
    }
    this.status = ShowStatus.Running;
    if (this.autoChangeName) {
      for (const character of this.playbook.characters) {
        const bot = this.characterBotMap.get(character.id);
        try {
          await bot.internal.setGroupCard(
            this.groupId,
            bot.selfId,
            character.name,
          );
        } catch (e) {
          return `Change name for ${bot.selfId} ${
            character.name
          } failed: ${e.toString()}`;
        }
      }
    }
    this.launchTime = moment();
    this.showInterval = setInterval(async () => {
      await this.showProcess();
    }, 1000);
    return null;
  }
  private async showProcess() {
    try {
      const exitShow = await this.playbook.play(this.step++);
      if (exitShow) {
        this.endShow('Show ended.');
      }
      return exitShow;
    } catch (e) {
      this.endShow(`Show errored: ${e.toString()}`);
      return true;
    }
  }
  endShow(message: string) {
    if (this.status !== ShowStatus.Running) {
      return false;
    }
    clearInterval(this.showInterval);
    this.showInterval = undefined;
    for (const character of this.playbook.characters) {
      character.playFun = undefined;
    }
    this.onFinish(message, this);
    return true;
  }
  getCharacterList(): string {
    return Array.from(this.characterBotMap.entries())
      .map(
        ([characterId, bot]) =>
          `${this.playbook.characters
            .find((c) => c.id === characterId)
            .getDisplayName()} => ${bot.selfId}`,
      )
      .join('\n');
  }
  getStatus(): string {
    return `群: ${this.groupId}\n开始时间: ${this.launchTime.format(
      'YYYY-MM-DD HH:mm:ss',
    )}\n人物: ${this.getCharacterList()}`;
  }
}
