import MemoryDriver from '@minatojs/driver-memory';
import { ModelDecorators } from '../src/decorators';
import { Database } from 'minato';
import { ModelRegistrar } from '../src/register';

interface Tables {
  dress: Dress;
  dress2: Dress2;
}
const decorators = new ModelDecorators<Tables>();

class DressProperty {
  @decorators.ModelField('string(8)')
  color: string;
  @decorators.ModelField('integer(7)')
  size: string;

  getProperty() {
    return `${this.color} ${this.size}`;
  }
}

@decorators.DefineModel('dress')
class Dress {
  @decorators.PrimaryGenerated()
  @decorators.ModelField('integer(11)')
  id: number;

  @decorators.Unique()
  @decorators.ModelField()
  name: string; // test if it can infer type

  getName() {
    return this.name;
  }

  @decorators.ModelField('integer(11)')
  @decorators.Foreign('dress', 'id')
  parentId: number;

  @decorators.ChildModel()
  properties: DressProperty;
}

@decorators.DefineModel('dress2')
class Dress2 {
  @decorators.Primary()
  @decorators.ModelField('string')
  owner: string;

  @decorators.ModelField('integer(11)')
  @decorators.Primary()
  size: number;
}

describe('Model test', () => {
  let model: Database<Tables>;

  beforeEach(async () => {
    model = new Database();
    await model.connect(MemoryDriver, {});
    const registrar = new ModelRegistrar(model);
    registrar.registerModel(Dress);
    registrar.registerModel(Dress2);
  });

  it('should register model fields', () => {
    const { dress } = model.tables;
    expect(dress.fields.id.type).toBe('integer');
    expect(dress.fields.id.length).toBe(11);
    expect(dress.fields.name.type).toBe('string');
    const colorConfig = dress.fields['properties.color'];
    expect(colorConfig.type).toBe('string');
    expect(colorConfig.length).toBe(8);
    const sizeConfig = dress.fields['properties.size'];
    expect(sizeConfig.type).toBe('integer');
    expect(sizeConfig.length).toBe(7);
  });

  it('should handle double primary', () => {
    const { dress2 } = model.tables;
    expect(dress2.primary).toStrictEqual(['owner', 'size']);
  })

  it('should register model extras', () => {
    const { dress } = model.tables;
    expect(dress.primary).toBe('id');
    expect(dress.unique[0][0]).toBe('name');
    expect(dress.foreign.parentId).toStrictEqual(['dress', 'id']);
    //expect(dress.internal['']).toEqual(Dress.prototype);
    //expect(dress.internal['properties.']).toEqual(DressProperty.prototype);
  });

  it('should make class instance', async () => {
    await model.upsert('dress', [
      {
        id: 777,
        name: 'Dress of Shigma',
        properties: {
          color: 'red',
          size: 10,
        },
      },
    ]);
    const [dress] = await model.get('dress', { id: 777 });
    expect(dress).toBeInstanceOf(Dress);
    expect(dress.id).toBe(777);
    expect(dress.properties.color).toBe('red');
    expect(dress.properties.size).toBe(10);
    expect(dress.getName()).toBe('Dress of Shigma');
    expect(dress.properties).toBeInstanceOf(DressProperty);
    expect(dress.properties.getProperty()).toBe('red 10');
  });
});
