import MemoryDriver from '@minatojs/driver-memory';
import { ModelDecorators } from '../src/decorators';
import { Database } from 'minato';
import { ModelRegistrar } from '../src/register';

interface User {
  id: string;
  dress: Dress;
  shirt: Shirt;
}

interface Tables {
  user: User;
}

const decorators = new ModelDecorators<Tables>();

class WearingPreference {
  @decorators.ModelField('string(8)')
  color: string;
  @decorators.ModelField('string(12)')
  shape: string;

  format() {
    return `${this.color} ${this.shape}`;
  }
}

class Wearing {
  @decorators.ModelField('string(3)')
  size: string;

  getSize() {
    return this.size;
  }

  @decorators.ChildModel()
  preference: WearingPreference;
}

class Dress extends Wearing {
  getDressSize() {
    return this.getSize();
  }
}
class Shirt extends Wearing {
  getShirtSize() {
    return this.getSize();
  }
}

describe('Model test', () => {
  let model: Database<Tables>;

  beforeEach(async () => {
    model = new Database();
    const registrar = new ModelRegistrar(model);
    model.extend(
      'user',
      {
        id: 'string(8)',
      },
      {
        primary: 'id',
      },
    );
    registrar.mixinModel('user', {
      dress: Dress,
      shirt: Shirt,
    });
    const database = new MemoryDriver(model, {});
    await database.start();
  });

  it('should register model fields', () => {
    const { user } = model.tables;
    expect(user.fields['dress.size'].type).toBe('string');
    expect(user.fields['dress.size'].length).toBe(3);
    expect(user.fields['shirt.size'].type).toBe('string');
    expect(user.fields['shirt.size'].length).toBe(3);
    expect(user.fields['dress.preference.color'].type).toBe('string');
    expect(user.fields['dress.preference.color'].length).toBe(8);
    expect(user.fields['shirt.preference.color'].type).toBe('string');
    expect(user.fields['shirt.preference.color'].length).toBe(8);
    expect(user.fields['dress.preference.shape'].type).toBe('string');
    expect(user.fields['dress.preference.shape'].length).toBe(12);
    expect(user.fields['shirt.preference.shape'].type).toBe('string');
    expect(user.fields['shirt.preference.shape'].length).toBe(12);

    expect(user.internal['dress.']).toEqual(Dress.prototype);
    expect(user.internal['shirt.']).toEqual(Shirt.prototype);
    expect(user.internal['dress.preference.']).toEqual(
      WearingPreference.prototype,
    );
    expect(user.internal['shirt.preference.']).toEqual(
      WearingPreference.prototype,
    );
  });

  it('should make class instance', async () => {
    await model.upsert('user', [
      {
        id: '111111',
        dress: {
          size: 'S',
          preference: {
            color: 'red',
            shape: 'round',
          },
        },
        shirt: {
          size: 'M',
          preference: {
            color: 'blue',
            shape: 'square',
          },
        },
      },
    ]);
    const [user] = await model.get('user', { id: '111111' });
    expect(user.dress).toBeInstanceOf(Dress);
    expect(user.dress.getDressSize()).toBe('S');
    expect(user.dress.preference).toBeInstanceOf(WearingPreference);
    expect(user.dress.preference.format()).toBe('red round');
    expect(user.shirt).toBeInstanceOf(Shirt);
    expect(user.shirt.getShirtSize()).toBe('M');
    expect(user.shirt.preference).toBeInstanceOf(WearingPreference);
    expect(user.shirt.preference.format()).toBe('blue square');
  });
});
