import { Database, Field, Flatten, Keys, Model } from 'cosmotype';
import { ModelClassType } from './def';
import { reflector } from './meta/meta';

class TableRegistrar<Tables, T = any> {
  constructor(private cls: ModelClassType<T>, private prefix = '') {}

  getTableName() {
    return reflector.get('ModelTableName', this.cls);
  }

  getFields(): Field.Extension<T> {
    const keys = reflector.getArray('ModelFieldKeys', this.cls);
    const result: Field.Extension<T> = {};
    for (const key of keys) {
      const field = reflector.get('ModelField', this.cls, key);
      if (field) {
        result[this.prefix + key] = field;
      }
    }
    return result;
  }

  getExtensions(): Model.Config<T> {
    const keys = reflector.getArray('ModelFieldKeys', this.cls);
    const result: Model.Config<T> = {};
    const primaryKeys: string[] = [];
    const uniqueMap = new Map<number | string, string[]>();
    for (const key of keys) {
      // primary keys
      const primary = reflector.get('ModelPrimaryKey', this.cls, key);
      if (primary) {
        primaryKeys.push(this.prefix + key);
        if (primary.autoIncrement) {
          result.autoInc = true;
        }
      }
      // foreign keys
      const foreign = reflector.get('ModelForeignKey', this.cls, key);
      if (foreign) {
        if (!result.foreign) {
          result.foreign = {};
        }
        result.foreign[this.prefix + key] = foreign;
      }
      // unique
      const uniqueEntries = reflector.getArray('ModelUnique', this.cls, key);
      for (const unique of uniqueEntries) {
        if (!uniqueMap.has(unique)) {
          uniqueMap.set(unique, []);
        }
        uniqueMap.get(unique).push(this.prefix + key);
      }
    }
    if (primaryKeys.length) {
      // eslint-disable-next-line @typescript-eslint/ban-ts-comment
      // @ts-ignore
      result.primary = primaryKeys;
    }
    const uniqueDefs = Array.from(uniqueMap.values());
    if (uniqueDefs.length) {
      // eslint-disable-next-line @typescript-eslint/ban-ts-comment
      // @ts-ignore
      result.unique = uniqueDefs;
    }
    return result;
  }

  private mergeFields(fieldDefs: Field.Extension[]) {
    let result: Field.Extension<T> = {};
    for (const fieldDef of fieldDefs) {
      result = { ...result, ...fieldDef };
    }
    return result;
  }

  private mergeExtensions(extDefs: Model.Config<T>[]) {
    const result: Model.Config<T> = {
      autoInc: extDefs.some((ext) => ext.autoInc),
    };
    for (const extDef of extDefs) {
      if (extDef.primary) {
        if (!result.primary) {
          // eslint-disable-next-line @typescript-eslint/ban-ts-comment
          // @ts-ignore
          result.primary = [];
        }
        // eslint-disable-next-line @typescript-eslint/ban-ts-comment
        // @ts-ignore
        result.primary = (result.primary as Keys<T, any>[]).concat(
          extDef.primary,
        );
      }
      if (extDef.unique) {
        if (!result.unique) {
          result.unique = [];
        }
        result.unique = result.unique.concat(extDef.unique);
      }
      if (extDef.foreign) {
        result.foreign = { ...(result.foreign || {}), ...extDef.foreign };
      }
    }
    return result;
  }

  getChildDict() {
    const keys = reflector.getArray('ChildModelKeys', this.cls);
    const result: { [K in keyof T]?: ModelClassType<T> } = {};
    for (const key of keys) {
      const child = reflector.get('ChildModel', this.cls, key);
      if (child) {
        result[key] = child;
      }
    }
    return result;
  }

  getInternal() {
    let internal = { [this.prefix]: this.cls.prototype };
    const childDict = this.getChildDict();
    for (const key in childDict) {
      const child = childDict[key];
      const prefix = this.prefix + key + '.';
      const childReg = new TableRegistrar<Tables>(child, prefix);
      internal = { ...internal, ...childReg.getInternal() };
    }
    return internal;
  }

  private getChildModelResults() {
    const children = this.getChildDict();
    const results: [Field.Extension, Model.Config][] = [];
    for (const key of Object.keys(children)) {
      const child = children[key];
      if (child) {
        const childRegistrar = new TableRegistrar<Tables>(
          child,
          this.prefix + key + '.',
        );
        results.push(childRegistrar.getModelResult());
      }
    }
    return {
      fields: results.map((r) => r[0]),
      extensions: results.map((r) => r[1]),
    };
  }

  getModelResult(): [Field.Extension<T>, Model.Config<T>] {
    const fields = this.getFields();
    const extensions = this.getExtensions();
    const childResults = this.getChildModelResults();
    return [
      this.mergeFields([fields, ...childResults.fields]),
      this.mergeExtensions([extensions, ...childResults.extensions]),
    ];
  }
}

export class ModelRegistrar<Tables> {
  constructor(private readonly model: Database<Tables>) {}

  registerModel(cls: ModelClassType<Flatten<Tables[Keys<Tables>]>>);
  registerModel(cls: ModelClassType) {
    const registrar = new TableRegistrar(cls);
    const tableName = registrar.getTableName();
    if (!tableName) {
      throw new Error(`Model of ${cls.name} is not defined`);
    }
    // eslint-disable-next-line @typescript-eslint/ban-ts-comment
    // @ts-ignore
    this.model.extend(tableName, ...registrar.getModelResult());
    Object.assign(
      this.model.tables[tableName].internal,
      registrar.getInternal(),
    );
  }

  mixinModel<K extends Keys<Tables>>(
    tableName: K,
    classDict: {
      [F in Keys<Tables[K]>]?: ModelClassType<Flatten<Tables[K][F]>>;
    },
  ) {
    for (const _key in classDict) {
      const key = _key as Keys<Tables[K]>;
      const cls = classDict[key];
      const registrar = new TableRegistrar<any>(cls, key + '.');
      const result = registrar.getModelResult();
      this.model.extend(tableName, ...result);
      Object.assign(
        this.model.tables[tableName].internal,
        registrar.getInternal(),
      );
    }
  }
}
