import {
  ChildModel,
  DefineModel,
  Foreign,
  ModelField,
  PrimaryGenerated,
  Unique,
} from '../src/decorators';
import { App } from 'koishi';
import { registerModel } from '../src/register';
import MemoryDatabase from '@koishijs/plugin-database-memory';

declare module 'koishi' {
  // eslint-disable-next-line @typescript-eslint/no-empty-interface
  interface Tables {
    dress: Dress;
  }
}

class DressProperty {
  @ModelField('string(8)')
  color: string;
  @ModelField('integer(7)')
  size: string;

  getProperty() {
    return `${this.color} ${this.size}`;
  }
}

@DefineModel('dress')
class Dress {
  @PrimaryGenerated()
  @ModelField('integer(11)')
  id: number;

  @Unique()
  @ModelField()
  name: string; // test if it can infer type

  getName() {
    return this.name;
  }

  @ModelField('integer(11)')
  @Foreign('dress', 'id')
  parentId: number;

  @ChildModel()
  properties: DressProperty;
}

describe('Model test', () => {
  let app: App;

  beforeEach(async () => {
    app = new App();
    registerModel(app, Dress);
    app.plugin(MemoryDatabase);
    await app.start();
  });

  it('should register model fields', () => {
    const { dress } = app.model.config;
    expect(dress.fields.id.type).toBe('integer');
    expect(dress.fields.id.length).toBe(11);
    expect(dress.fields.name.type).toBe('string');
    const colorConfig = dress.fields['properties.color'];
    expect(colorConfig.type).toBe('string');
    expect(colorConfig.length).toBe(8);
    const sizeConfig = dress.fields['properties.size'];
    expect(sizeConfig.type).toBe('integer');
    expect(sizeConfig.length).toBe(7);
  });

  it('should register model extras', () => {
    const { dress } = app.model.config;
    expect(dress.primary[0]).toBe('id');
    expect(dress.unique[0][0]).toBe('name');
    expect(dress.foreign.parentId).toStrictEqual(['dress', 'id']);
  });

  it('should make class instance', async () => {
    await app.database.upsert('dress', [
      {
        id: 777,
        name: 'Dress of Shigma',
        properties: {
          color: 'red',
          size: 10,
        },
      },
    ]);
    const [dress] = await app.database.get('dress', { id: 777 });
    expect(dress.id).toBe(777);
    expect(dress.getName()).toBe('Dress of Shigma');
    expect(dress.properties.getProperty()).toBe('red 10');
  });
});
