import { Context, Keys, Model } from 'koishi';
import { ModelClassType } from './def';
import { reflector } from './meta/meta';

class ModelRegistrar<T = any> {
  constructor(private cls: ModelClassType<T>, private prefix = '') {}

  getTableName() {
    return reflector.get('ModelTableName', this.cls);
  }

  getFields(): Model.Field.Extension<T> {
    const keys = reflector.getArray('ModelFieldKeys', this.cls);
    const result: Model.Field.Extension<T> = {};
    for (const key of keys) {
      const field = reflector.get('ModelField', this.cls, key);
      if (field) {
        result[this.prefix + key] = field;
      }
    }
    return result;
  }

  getExtensions(): Model.Extension<T> {
    const keys = reflector.getArray('ModelFieldKeys', this.cls);
    const result: Model.Extension<T> = {};
    const primaryKeys: string[] = [];
    const uniqueMap = new Map<number | string, string[]>();
    for (const key of keys) {
      // primary keys
      const primary = reflector.get('ModelPrimaryKey', this.cls, key);
      if (primary) {
        primaryKeys.push(this.prefix + key);
        if (primary.autoIncrement) {
          result.autoInc = true;
        }
      }
      // foreign keys
      const foreign = reflector.get('ModelForeignKey', this.cls, key);
      if (foreign) {
        if (!result.foreign) {
          result.foreign = {};
        }
        result.foreign[this.prefix + key] = foreign;
      }
      // unique
      const uniqueEntries = reflector.getArray('ModelUnique', this.cls, key);
      for (const unique of uniqueEntries) {
        if (!uniqueMap.has(unique)) {
          uniqueMap.set(unique, []);
        }
        uniqueMap.get(unique).push(this.prefix + key);
      }
    }
    if (primaryKeys.length) {
      // eslint-disable-next-line @typescript-eslint/ban-ts-comment
      // @ts-ignore
      result.primary = primaryKeys;
    }
    const uniqueDefs = Array.from(uniqueMap.values());
    if (uniqueDefs.length) {
      // eslint-disable-next-line @typescript-eslint/ban-ts-comment
      // @ts-ignore
      result.unique = uniqueDefs;
    }
    return result;
  }

  private mergeFields(fieldDefs: Model.Field.Extension[]) {
    let result: Model.Field.Extension<T> = {};
    for (const fieldDef of fieldDefs) {
      result = { ...result, ...fieldDef };
    }
    return result;
  }

  private mergeExtensions(extDefs: Model.Extension<T>[]) {
    const result: Model.Extension<T> = {
      autoInc: extDefs.some((ext) => ext.autoInc),
    };
    for (const extDef of extDefs) {
      if (extDef.primary) {
        if (!result.primary) {
          // eslint-disable-next-line @typescript-eslint/ban-ts-comment
          // @ts-ignore
          result.primary = [];
        }
        // eslint-disable-next-line @typescript-eslint/ban-ts-comment
        // @ts-ignore
        result.primary = (result.primary as Keys<T, any>[]).concat(
          extDef.primary,
        );
      }
      if (extDef.unique) {
        if (!result.unique) {
          result.unique = [];
        }
        result.unique = result.unique.concat(extDef.unique);
      }
      if (extDef.foreign) {
        result.foreign = { ...(result.foreign || {}), ...extDef.foreign };
      }
    }
    return result;
  }

  getChildDict() {
    const keys = reflector.getArray('ChildModelKeys', this.cls);
    const result: { [K in keyof T]?: ModelClassType<T> } = {};
    for (const key of keys) {
      const child = reflector.get('ChildModel', this.cls, key);
      if (child) {
        result[key] = child;
      }
    }
    return result;
  }

  getInternal() {
    let internal = { [this.prefix]: this.cls.prototype };
    const childDict = this.getChildDict();
    for (const key in childDict) {
      const child = childDict[key];
      const prefix = this.prefix + key + '.';
      const childReg = new ModelRegistrar(child, prefix);
      internal = { ...internal, ...childReg.getInternal() };
    }
    return internal;
  }

  private getChildModelResults() {
    const children = this.getChildDict();
    const results: [Model.Field.Extension, Model.Extension][] = [];
    for (const key of Object.keys(children)) {
      const child = children[key];
      if (child) {
        const childRegistrar = new ModelRegistrar(
          child,
          this.prefix + key + '.',
        );
        results.push(childRegistrar.getModelResult());
      }
    }
    return {
      fields: results.map((r) => r[0]),
      extensions: results.map((r) => r[1]),
    };
  }

  getModelResult(): [Model.Field.Extension<T>, Model.Extension<T>] {
    const fields = this.getFields();
    const extensions = this.getExtensions();
    const childResults = this.getChildModelResults();
    return [
      this.mergeFields([fields, ...childResults.fields]),
      this.mergeExtensions([extensions, ...childResults.extensions]),
    ];
  }
}

export function registerModel(
  ctx: Context,
  cls: { new (...args: any[]): any },
) {
  const registrar = new ModelRegistrar(cls);
  const tableName = registrar.getTableName();
  if (!tableName) {
    throw new Error(`Model of ${cls.name} is not defined`);
  }
  ctx.model.extend(tableName, ...registrar.getModelResult());
  ctx.model.config[tableName].internal = registrar.getInternal();
}
