import {
  CommandUsage,
  PutArg,
  PutObject,
  PutOption,
  PutValue,
  UseCommand,
  UseEvent,
  UseMiddleware,
} from '../src/decorators';
import { App, Command, Next, Session } from 'koishi';
import { Registrar } from '../src/register';
import { EventNameAndPrepend } from '../src/def';

class SkirtArg {
  @PutArg(0)
  count: number;

  @PutOption('color', '-c <color>')
  color: string;
}

class MyClass {
  @UseMiddleware()
  async onPing(session: Session, next: Next) {
    if (session.content === 'ping') {
      return 'pong';
    }
    return next();
  }

  @UseEvent('message')
  async onMessage(session: Session) {
    if (session.content === 'pang') {
      await session.send('peng');
    }
  }

  @UseCommand('echo', 'hi')
  @CommandUsage('foo')
  async onEcho(@PutOption('content', '-c <content>') content: string) {
    return `bot: ${content}`;
  }

  @UseCommand('count <count>')
  async onCount(@PutArg(0) count: number) {
    return `I have ${count} dresses.`;
  }

  @UseCommand('skirt')
  async onSkirt(@PutObject() arg: SkirtArg) {
    return `I have ${arg.count} ${arg.color} skirts.`;
  }

  @UseCommand('{{name}}')
  @CommandUsage('{{usage}}')
  async onGeneric(
    @PutOption('{{option}}', '-c <count>') count: number,
    @PutValue('{{name}}') name: string,
  ) {
    return `I have ${count} ${name}.`;
  }
}

const registrar = new Registrar(new MyClass());

describe('Register', () => {
  let app: App;

  beforeEach(async () => {
    app = new App();
    await app.start();
  });

  it('should register middleware', () => {
    const result = registrar.register(app, 'onPing');
    expect(result.type).toBe('middleware');
    expect(result.data).toEqual(false);
  });

  it('should register event', () => {
    const result = registrar.register(app, 'onMessage');
    expect(result.type).toBe('onEvent');
    expect((result.data as EventNameAndPrepend).name).toBe('message');
  });

  it('should register command and infer option type', () => {
    const result = registrar.register(app, 'onEcho');
    expect(result.type).toBe('command');
    const command: Command = result.result;
    expect(command._usage).toBe('foo');
    expect(command._options.content.name).toBe('content');
    expect(command._options.content.type).toBe('string');
    expect(command.execute({ options: { content: 'hello' } })).resolves.toBe(
      'bot: hello',
    );
  });

  it('should infer argument type', () => {
    const result = registrar.register(app, 'onCount');
    expect(result.type).toBe('command');
    const command: Command = result.result;
    expect(command._arguments[0].type).toBe('number');
    expect(command._arguments[0].required).toBe(true);
    expect(command.execute({ args: ['4'] })).resolves.toBe('I have 4 dresses.');
  });

  it('should work on class type', () => {
    const result = registrar.register(app, 'onSkirt');
    expect(result.type).toBe('command');
    const command: Command = result.result;
    expect(command._arguments[0].type).toBe('number');
    expect(command._options.color.type).toBe('string');
    expect(
      command.execute({ args: ['4'], options: { color: 'red' } }),
    ).resolves.toBe('I have 4 red skirts.');
  });

  it('should work on template', () => {
    const registrar = new Registrar(new MyClass(), undefined, {
      name: 'socks',
      usage: 'Socks count',
      option: 'count',
    });
    const result = registrar.register(app, 'onGeneric');
    expect(result.type).toBe('command');
    const command: Command = result.result;
    expect(command._usage).toBe('Socks count');
    expect(command.name).toBe('socks');
    expect(command._options.count.name).toBe('count');
    expect(command.execute({ options: { count: 7 } })).resolves.toBe(
      'I have 7 socks.',
    );
  });
});
