import { Argv, Command, Context, FieldCollector, Session, User } from 'koishi';
import {
  CommandArgDef,
  CommandOptionConfig,
  GenerateMappingStruct,
  KoishiCommandPutDef,
  MappingStruct,
  TemplateConfig,
} from '../../def';
import { MethodRegistry } from '../abstract-registry';
import {
  applyNativeTypeToArg,
  applyOptionToCommand,
  registerTemplate,
  renderObject,
} from '../../utility';
import { Metadata } from '../../meta/metadata.decorators';
import { reflector } from '../../meta/meta-fetch';

// eslint-disable-next-line @typescript-eslint/no-namespace
export namespace CommandPut {
  export interface ConfigMap {
    args: void;
    arg: CommandArgDef;
    argv: void;
    argvField: keyof Argv;
    option: CommandOptionConfig;
    user: FieldCollector<'user'>;
    channel: FieldCollector<'channel'>;
    guild: FieldCollector<'channel'>;
    username: boolean;
    sessionField: keyof Session;
    renderer: string | undefined;
    template: TemplateConfig;
    typeClass: void;
    value: string;
  }

  export type Config<K extends keyof ConfigMap = keyof ConfigMap> =
    MappingStruct<ConfigMap, K>;

  export const preRegistry = new MethodRegistry<
    ConfigMap,
    void,
    // eslint-disable-next-line @typescript-eslint/ban-types
    [Command, Context, Function, any]
  >();

  preRegistry.extend('option', (data, cmd, ctx, nativeType) =>
    applyOptionToCommand(ctx, cmd, data, nativeType),
  );

  preRegistry.extend('arg', (data, cmd, ctx, nativeType) => {
    let arg = cmd._arguments[data.index];
    if (!arg) {
      arg = {};
      cmd._arguments[data.index] = arg;
    }
    applyNativeTypeToArg(arg, nativeType);
    if (data.decl) {
      Object.assign(arg, data.decl);
    }
  });

  preRegistry.extend('user', (data, cmd) => {
    if (data) {
      cmd.userFields(data);
    }
  });

  preRegistry.extend('channel', (data, cmd) => {
    if (data) {
      cmd.channelFields(data);
    }
  });

  preRegistry.extend('username', (data, cmd) => {
    if (data) {
      cmd.userFields(['name']);
    }
  });

  preRegistry.extend('template', (data, cmd, ctx) =>
    registerTemplate(data, ctx, cmd),
  );

  preRegistry.extend('typeClass', (data, cmd, ctx, nativeType, view) => {
    const keys = reflector.getArray('KoishiPutClassFieldKeys', nativeType);
    for (const key of keys) {
      const meta = reflector.get('KoishiPutClassField', nativeType, key);
      if (!meta) continue;
      const propertyNativeType = Reflect.getMetadata(
        'design:type',
        nativeType.prototype,
        key,
      );
      preRegistry.execute(
        renderObject(meta, view),
        cmd,
        ctx,
        propertyNativeType,
        view,
      );
    }
  });

  export const registry = new MethodRegistry<
    ConfigMap,
    any,
    // eslint-disable-next-line @typescript-eslint/ban-types
    [Argv, any[], Function, any]
  >();

  registry.extend('value', (data) => data);
  registry.extend('args', (data, argv, args) => args);
  registry.extend('arg', (data, argv, args) => args[data.index]);
  registry.extend('argv', (data, argv, args) => argv);
  registry.extend('argvField', (data, argv, args) => argv[data]);
  registry.extend('option', (data, argv, args) => argv.options[data.name]);
  registry.extend('user', (data, argv, args) => argv.session.user);
  registry.extend('channel', (data, argv, args) => argv.session.channel);
  registry.extend('guild', (data, argv, args) => argv.session.guild);
  registry.extend('username', (useDatabase, argv, args) => {
    if (useDatabase) {
      const user = argv.session.user as User.Observed<'name'>;
      if (user?.name) {
        return user?.name;
      }
    }
    return (
      argv.session.author?.nickname ||
      argv.session.author?.username ||
      argv.session.userId
    );
  });
  registry.extend('sessionField', (data, argv, args) => argv.session[data]);

  registry.extend('renderer', (data, argv, args) =>
    data
      ? // eslint-disable-next-line @typescript-eslint/ban-types
        (params: object) => argv.session.text(data, params)
      : // eslint-disable-next-line @typescript-eslint/ban-types
        (path: string, params: object) => argv.session.text(path, params),
  );
  registry.extend(
    'template',
    // eslint-disable-next-line @typescript-eslint/ban-types
    (data, argv, args) => (params: object) =>
      argv.session.text(`.${data.name}`, params),
  );
  registry.extend(
    'typeClass',
    (data, argv, args, nativeType: { new (): any }, view) => {
      const keys = reflector.getArray('KoishiPutClassFieldKeys', nativeType);
      const obj = new nativeType();
      for (const key of keys) {
        const meta = reflector.get('KoishiPutClassField', nativeType, key);
        if (!meta) continue;
        const propertyNativeType = Reflect.getMetadata(
          'design:type',
          nativeType.prototype,
          key,
        );
        obj[key] = registry.execute(
          renderObject(meta, view),
          argv,
          args,
          propertyNativeType,
          view,
        );
      }
      return obj;
    },
  );

  export function decorate<T extends keyof ConfigMap>(
    type: T,
    data?: ConfigMap[T],
  ): ParameterDecorator & PropertyDecorator {
    return (obj, key: string, index?: number) => {
      const def = GenerateMappingStruct(type, data);
      if (typeof index === 'number') {
        // As a parameter decorator
        const objClass = obj.constructor;
        const list: Config<T>[] =
          Reflect.getMetadata(KoishiCommandPutDef, objClass, key) || [];
        list[index] = def;
        Reflect.defineMetadata(KoishiCommandPutDef, list, objClass, key);
      } else {
        // As a property decorator
        Metadata.set(
          'KoishiPutClassField',
          def,
          'KoishiPutClassFieldKeys',
        )(obj, key);
      }
    };
  }
}
