import {
  Argv,
  Command,
  Context,
  Dict,
  EventMap,
  Plugin,
  Selection,
  I18n,
  Awaitable,
} from 'koishi';
import type { DefaultContext, DefaultState, ParameterizedContext } from 'koa';
import type { RouterParamContext } from '@koa/router';
import { CommandPut } from '../registry/registries/command-put';

export interface Type<T = any> extends Function {
  new (...args: any[]): T;
}

export interface ContextSelector {
  select?: Selection;
  useSelector?: OnContextFunction;
}

export type PluginOptions<T extends Plugin> = boolean | Plugin.Config<T>;

export interface PluginDefinitionExact<T extends Plugin>
  extends ContextSelector {
  plugin: T;
  options?: boolean | PluginOptions<T>;
}

export interface PluginDefinitionName extends ContextSelector {
  plugin: string;
  options?: any;
}

export type PluginDefinition<T extends Plugin = any> =
  | PluginDefinitionExact<T>
  | PluginDefinitionName;

export function PluginDef(
  name: string,
  options?: any,
  select?: Selection,
): PluginDefinitionName;
export function PluginDef<T extends Plugin>(
  plugin: T,
  options?: PluginOptions<T>,
  select?: Selection,
): PluginDefinitionExact<T>;
export function PluginDef<T extends Plugin>(
  plugin: T,
  options?: PluginOptions<T>,
  select?: Selection,
): PluginDefinition<T> {
  return { plugin, options, select };
}

export interface CommonEventNameAndPrepend<T extends keyof any> {
  name: T;
  prepend?: boolean;
}

export type EventName = keyof EventMap;
export type EventNameAndPrepend = CommonEventNameAndPrepend<EventName>;

type OmitSubstring<
  S extends string,
  T extends string,
> = S extends `${infer L}${T}${infer R}` ? `${L}${R}` : never;
export type BeforeEventName = OmitSubstring<EventName & string, 'before-'>;
export type BeforeEventNameAndPrepend =
  CommonEventNameAndPrepend<BeforeEventName>;

export type ContextFunction<T, C extends Context = Context> = (ctx: C) => T;
export type OnContextFunction<C extends Context = Context> = ContextFunction<
  C,
  C
>;

export interface MappingStruct<
  T extends Record<string | number | symbol, any>,
  K extends keyof T,
> {
  type: K;
  data?: T[K];
}

export function GenerateMappingStruct<
  T extends Record<string | number | symbol, any>,
  K extends keyof T,
>(type: K, data?: T[K]): MappingStruct<T, K> {
  return {
    type,
    data,
  };
}

// Command stuff

export interface CommandRegisterConfig<D extends string = string> {
  def: D;
  desc?: string;
  config?: CommandConfigExtended;
  putOptions?: CommandPut.Config[];
  // eslint-disable-next-line @typescript-eslint/ban-types
  paramTypes: Function[];
}

export interface CommandConfigExtended extends Command.Config {
  empty?: boolean;
}

export interface CommandOptionConfig {
  name: string;
  desc: string;
  config?: CommandOptionConfigWithDescription;
}

export type CommandDefinitionFun = (
  cmd: Command,
  ctx: Context,
  renderer: ParamRenderer,
) => Command;

export interface KoishiRouteDef {
  path: string;
  method:
    | 'get'
    | 'post'
    | 'put'
    | 'delete'
    | 'patch'
    | 'options'
    | 'head'
    | 'all';
}

export type KoaContext = ParameterizedContext<
  DefaultState,
  DefaultContext & RouterParamContext<DefaultState, DefaultContext>,
  any
>;

// eslint-disable-next-line @typescript-eslint/ban-types
export type Renderer<T = any> = (params?: T) => string;
// eslint-disable-next-line @typescript-eslint/ban-types
export type CRenderer = (path: string, params?: object) => string;

export interface TemplateConfig {
  name: string;
  text: Dict<string>;
}

export type TopLevelActionDef = (
  ctx: Context,
  obj: any,
  renderer: ParamRenderer,
) => void;

export interface CommandOptionConfigWithDescription extends Argv.OptionConfig {
  description?: string | Dict<string>;
}

export interface CommandLocaleDef extends I18n.Store {
  description?: string;
  options?: Dict<string>;
  usage?: string;
  examples?: string;
  messages?: I18n.Store;
}

export interface CommandArgDef {
  index: number;
  decl?: Argv.Declaration;
}

export type ParamRenderer = <T>(v: T) => T;

export type TypedMethodDecorator<F extends (...args: any[]) => any> = <
  T extends F,
>(
  // eslint-disable-next-line @typescript-eslint/ban-types
  target: Object,
  propertyKey: string | symbol,
  descriptor: TypedPropertyDescriptor<T>,
) => void;

export type FunctionParam<F extends (...args: any[]) => any> = F extends (
  ...args: infer R
) => any
  ? R
  : never;
export type FunctionReturn<F extends (...args: any[]) => any> = F extends (
  ...args: any[]
) => infer R
  ? R
  : never;

export type PickEventFunction<M, K extends keyof M> = M[K] extends (
  ...args: any[]
) => any
  ? (...args: FunctionParam<M[K]>) => Awaitable<FunctionReturn<M[K]>>
  : M[K];
