import {
  BeforeEventName,
  CommandConfigExtended,
  CommandDefinitionFun,
  CommandLocaleDef,
  CommandOptionConfigWithDescription,
  EventName,
  KoishiCommandDefinition,
  KoishiCommandPutDef,
  KoishiOnContextScope,
  OnContextFunction,
  PickEventFunction,
  PluginDefinition,
  TopLevelActionDef,
  TypedMethodDecorator,
} from '../def';
import 'reflect-metadata';
import {
  Argv,
  Command,
  Dict,
  FieldCollector,
  Selection,
  Session,
  I18n,
  Awaitable,
  Middleware,
} from 'koishi';
import { Metadata } from '../meta/metadata.decorators';
import { CommandPut, DoRegister } from '../registry';
import {
  adaptLocaleDict,
  applyOptionToCommand,
  registerTemplate,
} from '../utility';
import { EventMap, BeforeEventMap } from 'koishi';

// Register method

export const UseMiddleware = (prepend = false) =>
  DoRegister.middleware(prepend);
export const UseEvent = <K extends EventName>(
  name: K,
  prepend = false,
): TypedMethodDecorator<PickEventFunction<EventMap, K>> =>
  DoRegister.event({ name, prepend });
export const UseBeforeEvent = <K extends BeforeEventName>(
  name: K,
  prepend = false,
): TypedMethodDecorator<PickEventFunction<BeforeEventMap, K>> =>
  DoRegister.beforeEvent({ name, prepend });
export const UsePlugin = () => DoRegister.plugin();

export function UseCommand<D extends string>(
  def: D,
  config?: CommandConfigExtended,
): TypedMethodDecorator<
  (...args: any[]) => Awaitable<string | void | undefined>
>;
export function UseCommand<D extends string>(
  def: D,
  desc: string,
  config?: CommandConfigExtended,
): TypedMethodDecorator<
  (...args: any[]) => Awaitable<string | void | undefined>
>;
export function UseCommand(
  def: string,
  ...args: [CommandConfigExtended?] | [string, CommandConfigExtended?]
): TypedMethodDecorator<
  (...args: any[]) => Awaitable<string | void | undefined>
> {
  const desc = typeof args[0] === 'string' ? (args.shift() as string) : '';
  const config = args[0] as CommandConfigExtended;
  return (obj, key: string, des) => {
    const putOptions: CommandPut.Config<keyof CommandPut.ConfigMap>[] =
      Reflect.getMetadata(KoishiCommandPutDef, obj.constructor, key) ||
      undefined;
    // eslint-disable-next-line @typescript-eslint/ban-types
    const paramTypes: Function[] = Reflect.getMetadata(
      'design:paramtypes',
      obj,
      key,
    );
    const metadataDec = DoRegister.command({
      def,
      desc,
      config,
      putOptions,
      paramTypes,
    });
    return metadataDec(obj, key, des);
  };
}

export const UseFormatter = (name: string) => DoRegister.formatter(name);
export const UsePreset = (name: string) => DoRegister.preset(name);
export const UseInterval = (interval: number) => DoRegister.interval(interval);

// Context scopes

export const OnContext = (
  ctxFun: OnContextFunction,
): MethodDecorator & ClassDecorator =>
  Metadata.append(KoishiOnContextScope, ctxFun);

export const OnAnywhere = () => OnContext((ctx) => ctx.any());

export const OnNowhere = () => OnContext((ctx) => ctx.never());

export const OnUser = (...values: string[]) =>
  OnContext((ctx) => ctx.user(...values));

export const OnSelf = (...values: string[]) =>
  OnContext((ctx) => ctx.self(...values));

export const OnGuild = (...values: string[]) =>
  OnContext((ctx) => ctx.guild(...values));

export const OnChannel = (...values: string[]) =>
  OnContext((ctx) => ctx.channel(...values));

export const OnPlatform = (...values: string[]) =>
  OnContext((ctx) => ctx.platform(...values));

export const OnPrivate = (...values: string[]) =>
  OnContext((ctx) => ctx.private(...values));

export const OnSelection = (selection: Selection) =>
  OnContext((ctx) => ctx.select(selection));

// Command definition

export const CommandDef = (
  def: CommandDefinitionFun,
): MethodDecorator & ClassDecorator =>
  Metadata.append(KoishiCommandDefinition, def);

export const CommandUse = <T extends Command, R extends any[]>(
  callback: (command: Command, ...args: R) => T,
  ...args: R
) => CommandDef((cmd, ctx, r) => callback(cmd, ...args));

export const CommandLocale = (locale: string, def: CommandLocaleDef) =>
  CommandDef((cmd, ctx, r) => {
    ctx.i18n.define(r(locale), `commands.${cmd.name}`, r(def));
    return cmd;
  });

export const CommandDescription = (desc: string | Dict<string>) => {
  return CommandDef((cmd, ctx, r) => {
    for (const localData of Object.entries(adaptLocaleDict(r(desc)))) {
      const [locale, text] = localData;
      ctx.i18n.define(locale, `commands.${cmd.name}.description`, text);
    }
    return cmd;
  });
};

export const CommandAlias = (...names: string[]) =>
  CommandDef((cmd, ctx, r) => cmd.alias(...r(names)));

export const CommandShortcut = (
  name: string | RegExp,
  config: Command.Shortcut = {},
) => CommandDef((cmd, ctx, r) => cmd.shortcut(r(name), r(config)));

export const CommandUsage = (text: Command.Usage) =>
  CommandDef((cmd, ctx, r) => cmd.usage(r(text)));

export const CommandExample = (text: string) =>
  CommandDef((cmd, ctx, r) => cmd.example(r(text)));

export const CommandOption = (
  name: string,
  desc: string,
  config: CommandOptionConfigWithDescription = {},
) =>
  CommandDef((cmd, ctx, r) =>
    applyOptionToCommand(ctx, cmd, r({ name, desc, config })),
  );

export const CommandUserFields = (fields: FieldCollector<'user'>) =>
  CommandDef((cmd, ctx, r) => cmd.userFields(r(fields)));

export const CommandChannelFields = (fields: FieldCollector<'channel'>) =>
  CommandDef((cmd, ctx, r) => cmd.channelFields(r(fields)));

export const CommandBefore = (callback: Command.Action, append = false) =>
  CommandDef((cmd, ctx, r) => cmd.before(callback, append));

export const CommandAction = (callback: Command.Action, prepend = false) =>
  CommandDef((cmd, ctx, r) => cmd.action(callback, prepend));

export const CommandTemplate = (name: string, text: string | Dict<string>) =>
  CommandDef((cmd, ctx, r) => {
    registerTemplate(
      { name: r(name), text: adaptLocaleDict(r(text)) },
      ctx,
      cmd,
    );
    return cmd;
  });

// Command put config

export const PutValue = (value: string) => CommandPut.decorate('value', value);
export const PutArgv = (field?: keyof Argv) =>
  field ? CommandPut.decorate('argvField', field) : CommandPut.decorate('argv');
export const PutSession = (field?: keyof Session) =>
  field ? CommandPut.decorate('sessionField', field) : PutArgv('session');
export const PutArg = (index: number, decl?: Argv.Declaration) =>
  CommandPut.decorate('arg', { index, decl });
export const PutArgs = () => CommandPut.decorate('args');
export const PutOption = (
  name: string,
  desc: string,
  config: CommandOptionConfigWithDescription = {},
) => CommandPut.decorate('option', { name, desc, config });

export const PutUser = (field: FieldCollector<'user'>) =>
  CommandPut.decorate('user', field);

export const PutChannel = (field: FieldCollector<'channel'>) =>
  CommandPut.decorate('channel', field);

export const PutGuild = (field: FieldCollector<'channel'>) =>
  CommandPut.decorate('guild', field);

export const PutUserName = (useDatabase = true) =>
  CommandPut.decorate('username', useDatabase);

export const PutUserId = () => PutSession('userId');
export const PutGuildId = () => PutSession('guildId');
export const PutGuildName = () => PutSession('guildName');
export const PutChannelId = () => PutSession('channelId');
export const PutChannelName = () => PutSession('channelName');
export const PutSelfId = () => PutSession('selfId');
export const PutBot = () => PutSession('bot');
export const PutNext = () => PutArgv('next');
export const PutRenderer = (path: string) =>
  CommandPut.decorate('renderer', path);
export const PutCommonRenderer = () => CommandPut.decorate('renderer');
export const PutTemplate = (name: string, text: string | Dict<string>) =>
  CommandPut.decorate('template', {
    name,
    text: adaptLocaleDict(text),
  });
export const PutObject = () => CommandPut.decorate('typeClass');

export const TopLevelAction = (action: TopLevelActionDef): ClassDecorator =>
  Metadata.append('KoishiTopLevelAction', action);

export const DefineTemplate = (name: string, text: string | Dict<string>) =>
  TopLevelAction((ctx, obj, r) =>
    registerTemplate({ name, text: adaptLocaleDict(r(text)) }, ctx),
  );

export function DefineLocale(locale: string, dict: I18n.Store): ClassDecorator;
export function DefineLocale(
  locale: string,
  key: string,
  value: I18n.Node,
): ClassDecorator;
export function DefineLocale(
  locale: string,
  ...args: [I18n.Store] | [string, I18n.Node]
): ClassDecorator {
  return TopLevelAction((ctx, obj, r) =>
    // eslint-disable-next-line @typescript-eslint/ban-ts-comment
    // @ts-ignore
    ctx.i18n.define(r(locale), ...r(args)),
  );
}
