import {
  CommandUsage,
  PutArg,
  PutObject,
  PutOption,
  UseCommand,
  UseEvent,
  UseMiddleware,
} from '../src/decorators';
import { App, Command, Next, Session } from 'koishi';
import { Registrar } from '../src/register';
import { EventNameAndPrepend } from '../src/def';

class SkirtArg {
  @PutArg(0)
  count: number;

  @PutOption('color', '-c <color>')
  color: string;
}

class MyClass {
  @UseMiddleware()
  async onPing(session: Session, next: Next) {
    if (session.content === 'ping') {
      return 'pong';
    }
    return next;
  }

  @UseEvent('message')
  async onMessage(session: Session) {
    if (session.content === 'pang') {
      return session.send('peng');
    }
  }

  @UseCommand('echo', 'hi')
  @CommandUsage('foo')
  async onEcho(@PutOption('content', '-c <content>') content: string) {
    return `bot: ${content}`;
  }

  @UseCommand('count <count>')
  async onCount(@PutArg(0) count: number) {
    return `I have ${count} dresses.`;
  }

  @UseCommand('skirt')
  async onSkirt(@PutObject() arg: SkirtArg) {
    return `I have ${arg.count} ${arg.color} skirts.`;
  }
}

const registrar = new Registrar(new MyClass());

describe('Register', () => {
  let app: App;

  beforeEach(async () => {
    app = new App();
    await app.start();
  });

  it('should register middleware', () => {
    const result = registrar.register(app, 'onPing');
    expect(result.type).toBe('middleware');
    expect(result.data).toEqual(false);
  });

  it('should register event', () => {
    const result = registrar.register(app, 'onMessage');
    expect(result.type).toBe('onEvent');
    expect((result.data as EventNameAndPrepend).name).toBe('message');
  });

  it('should register command and infer option type', () => {
    const result = registrar.register(app, 'onEcho');
    expect(result.type).toBe('command');
    const command: Command = result.result;
    expect(command._usage).toBe('foo');
    expect(command._options.content.name).toBe('content');
    expect(command._options.content.type).toBe('string');
    expect(command.execute({ options: { content: 'hello' } })).resolves.toBe(
      'bot: hello',
    );
  });

  it('should infer argument type', () => {
    const result = registrar.register(app, 'onCount');
    expect(result.type).toBe('command');
    const command: Command = result.result;
    expect(command._arguments[0].type).toBe('number');
    expect(command._arguments[0].required).toBe(true);
    expect(command.execute({ args: ['4'] })).resolves.toBe('I have 4 dresses.');
  });

  it('should work on class type', () => {
    const result = registrar.register(app, 'onSkirt');
    expect(result.type).toBe('command');
    const command: Command = result.result;
    expect(command._arguments[0].type).toBe('number');
    expect(command._options.color.type).toBe('string');
    expect(
      command.execute({ args: ['4'], options: { color: 'red' } }),
    ).resolves.toBe('I have 4 red skirts.');
  });
});
