import {
  CommandUsage,
  PutOption,
  UseCommand,
  UseEvent,
  UseMiddleware,
} from '../src/decorators/decorators';
import { App, Command, Next, Session } from 'koishi';
import { Registrar } from '../src/register';
import { EventNameAndPrepend } from '../src/def';

class MyClass {
  @UseMiddleware()
  async onPing(session: Session, next: Next) {
    if (session.content === 'ping') {
      return 'pong';
    }
    return next;
  }

  @UseEvent('message')
  async onMessage(session: Session) {
    if (session.content === 'pang') {
      return session.send('peng');
    }
  }

  @UseCommand('echo', 'hi')
  @CommandUsage('foo')
  async onEcho(@PutOption('content', '-c <content:string>') content: string) {
    return `bot: ${content}`;
  }
}

const registrar = new Registrar(new MyClass());

describe('Register', () => {
  let app: App;

  beforeEach(async () => {
    app = new App();
    await app.start();
  });

  it('should register middleware', () => {
    const result = registrar.register(app, 'onPing');
    expect(result.type).toBe('middleware');
    expect(result.data).toEqual(false);
  });

  it('should register event', () => {
    const result = registrar.register(app, 'onMessage');
    expect(result.type).toBe('onEvent');
    expect((result.data as EventNameAndPrepend).name).toBe('message');
  });

  it('should register command', () => {
    const result = registrar.register(app, 'onEcho');
    expect(result.type).toBe('command');
    const command: Command = result.result;
    expect(command._usage).toBe('foo');
    expect(command._options.content.name).toBe('content');
    expect(command.execute({ options: { content: 'hello' } })).resolves.toBe(
      'bot: hello',
    );
  });
});
