import { MethodMap, MethodRegistry } from '../abstract-registry';
import { Argv, Awaitable, Context, MaybeArray } from 'koishi';
import {
  BeforeEventNameAndPrepend,
  CommandRegisterConfig,
  EventNameAndPrepend,
  KoishiCommandDefinition,
  KoishiDoRegister,
  KoishiDoRegisterKeys,
  KoishiRouteDef,
  MappingStruct,
  PluginDefinition,
} from '../../def';
import { Metadata } from '../../meta/metadata.decorators';
import { reflector } from '../../meta/meta-fetch';
import { CommandPut } from './command-put';
import { applySelector } from '../../utility';

// eslint-disable-next-line @typescript-eslint/no-namespace
export namespace DoRegister {
  export function decorate<K extends keyof ConfigMap>(
    config: Config<K>,
  ): MethodDecorator {
    return Metadata.set(KoishiDoRegister, config, KoishiDoRegisterKeys);
  }

  class SpecificRegistry extends MethodRegistry<
    ConfigMap,
    any,
    [Context, any, string, any | undefined]
  > {
    define<K extends keyof ConfigMap>(
      name: K,
      method: MethodMap<ConfigMap, any, [Context, any, string, ...any[]]>[K],
    ) {
      this.extend(name, method);
      return (config: ConfigMap[K]) => decorate({ type: name, data: config });
    }
  }

  export interface ConfigMap {
    middleware: boolean; // prepend
    onEvent: EventNameAndPrepend;
    beforeEvent: BeforeEventNameAndPrepend;
    plugin: void;
    command: CommandRegisterConfig;
    route: KoishiRouteDef;
    ws: MaybeArray<string | RegExp>;
  }

  export type Config<K extends keyof ConfigMap = keyof ConfigMap> =
    MappingStruct<ConfigMap, K>;

  export const registry = new SpecificRegistry();

  export const middleware = registry.define(
    'middleware',
    (prepend, ctx, obj, key) =>
      ctx.middleware((session, next) => obj[key](session, next), prepend),
  );

  export const event = registry.define('onEvent', (data, ctx, obj, key) =>
    ctx.on(
      data.name,
      (session, ...args) => obj[key](session, ...args),
      data.prepend,
    ),
  );

  export const beforeEvent = registry.define(
    'beforeEvent',
    (data, ctx, obj, key) =>
      ctx.before(
        data.name,
        (session, ...args) => obj[key](session, ...args),
        data.prepend,
      ),
  );

  export const route = registry.define('route', (data, ctx, obj, key) => {
    const path = data.path.startsWith('/') ? data.path : `/${data.path}`;
    return ctx.router[data.method](path, (koaCtx, next) =>
      obj[key](koaCtx, next),
    );
  });

  export const ws = registry.define('ws', (data, ctx, obj, key) =>
    ctx.router.ws(data, (socket, request) => obj[key](socket, request)),
  );

  function applyInnerPluginDef(
    ctx: Context,
    key: string,
    pluginDef: PluginDefinition<any>,
  ) {
    const pluginCtx = applySelector(ctx, pluginDef);
    if (pluginDef == null) {
      return;
    }
    if (!pluginDef || !pluginDef.plugin) {
      throw new Error(`Invalid plugin from method ${key}.`);
    }
    pluginCtx.plugin(pluginDef.plugin, pluginDef.options);
  }

  export const plugin = registry.define('plugin', (_, ctx, obj, key) => {
    const pluginDescMayBeProm: Awaitable<PluginDefinition<any>> = obj[key]();
    if (pluginDescMayBeProm instanceof Promise) {
      pluginDescMayBeProm.then((pluginDef) => {
        applyInnerPluginDef(ctx, key, pluginDef);
      });
    } else {
      applyInnerPluginDef(ctx, key, pluginDescMayBeProm);
    }
  });

  export const command = registry.define(
    'command',
    (data, ctx, obj, key, extraObj) => {
      let command = ctx.command(data.def, data.desc, data.config);
      const commandDefs = reflector.getProperty(
        KoishiCommandDefinition,
        obj,
        key,
        extraObj,
      );
      for (const commandDef of commandDefs) {
        command = commandDef(command) || command;
      }
      if (!data.config?.empty) {
        if (!data.putOptions) {
          command.action((argv: Argv, ...args: any[]) =>
            obj[key](argv, ...args),
          );
        } else {
          for (const putOption of data.putOptions) {
            CommandPut.preRegistry.execute(putOption, command);
          }
          command.action((argv: Argv, ...args: any[]) => {
            const params = data.putOptions.map((o) =>
              CommandPut.registry.execute(o, argv, args),
            );
            return obj[key](...params);
          });
        }
      }
      return command;
    },
  );
}
