import { Argv, Command, FieldCollector, Session, User } from 'koishi';
import {
  CommandOptionConfig,
  GenerateMappingStruct,
  KoishiCommandPutDef,
  MappingStruct,
} from '../../def';
import { MethodRegistry } from '../abstract-registry';

// eslint-disable-next-line @typescript-eslint/no-namespace
export namespace CommandPut {
  export interface ConfigMap {
    args: void;
    arg: number;
    argv: void;
    argvField: keyof Argv;
    option: CommandOptionConfig;
    user: FieldCollector<'user'>;
    channel: FieldCollector<'channel'>;
    guild: FieldCollector<'channel'>;
    username: boolean;
    sessionField: keyof Session;
  }

  export type Config<K extends keyof ConfigMap = keyof ConfigMap> =
    MappingStruct<ConfigMap, K>;

  export const preRegistry = new MethodRegistry<ConfigMap, void, [Command]>();

  preRegistry.extend('option', (data, cmd) =>
    cmd.option(data.name, data.desc, data.config),
  );

  preRegistry.extend('user', (data, cmd) => {
    if (data) {
      cmd.userFields(data);
    }
  });

  preRegistry.extend('channel', (data, cmd) => {
    if (data) {
      cmd.channelFields(data);
    }
  });

  preRegistry.extend('username', (data, cmd) => {
    if (data) {
      cmd.userFields(['name']);
    }
  });

  export const registry = new MethodRegistry<ConfigMap, any, [Argv, any[]]>();

  registry.extend('args', (data, argv, args) => args);
  registry.extend('arg', (data, argv, args) => args[data]);
  registry.extend('argv', (data, argv, args) => argv);
  registry.extend('argvField', (data, argv, args) => argv[data]);
  registry.extend('option', (data, argv, args) => argv.options[data.name]);
  registry.extend('user', (data, argv, args) => argv.session.user);
  registry.extend('channel', (data, argv, args) => argv.session.channel);
  registry.extend('guild', (data, argv, args) => argv.session.guild);
  registry.extend('username', (useDatabase, argv, args) => {
    if (useDatabase) {
      const user = argv.session.user as User.Observed<'name'>;
      if (user?.name) {
        return user?.name;
      }
    }
    return (
      argv.session.author?.nickname ||
      argv.session.author?.username ||
      argv.session.userId
    );
  });
  registry.extend('sessionField', (data, argv, args) => argv.session[data]);

  export function decorate<T extends keyof ConfigMap>(
    type: T,
    data?: ConfigMap[T],
  ): ParameterDecorator {
    return (obj, key: string, index) => {
      const objClass = obj.constructor;
      const list: Config<T>[] =
        Reflect.getMetadata(KoishiCommandPutDef, objClass, key) || [];
      list[index] = GenerateMappingStruct(type, data);
      Reflect.defineMetadata(KoishiCommandPutDef, list, objClass, key);
    };
  }
}
