import { Argv, Command, Context, EventMap, Plugin, Selection } from 'koishi';
import type { DefaultContext, DefaultState, ParameterizedContext } from 'koa';
import type { RouterParamContext } from '@koa/router';
import { CommandPut } from '../registry/registries/command-put';

export interface Type<T = any> extends Function {
  new (...args: any[]): T;
}

export interface ContextSelector {
  select?: Selection;
  useSelector?: OnContextFunction;
}

export type PluginOptions<T extends Plugin> = boolean | Plugin.Config<T>;

export interface PluginDefinitionExact<T extends Plugin>
  extends ContextSelector {
  plugin: T;
  options?: boolean | PluginOptions<T>;
}

export interface PluginDefinitionName extends ContextSelector {
  plugin: string;
  options?: any;
}

export type PluginDefinition<T extends Plugin = any> =
  | PluginDefinitionExact<T>
  | PluginDefinitionName;

export function PluginDef(
  name: string,
  options?: any,
  select?: Selection,
): PluginDefinitionName;
export function PluginDef<T extends Plugin>(
  plugin: T,
  options?: PluginOptions<T>,
  select?: Selection,
): PluginDefinitionExact<T>;
export function PluginDef<T extends Plugin>(
  plugin: T,
  options?: PluginOptions<T>,
  select?: Selection,
): PluginDefinition<T> {
  return { plugin, options, select };
}

export interface CommonEventNameAndPrepend<T extends keyof any> {
  name: T;
  prepend?: boolean;
}

export type EventName = keyof EventMap;
export type EventNameAndPrepend = CommonEventNameAndPrepend<EventName>;

type OmitSubstring<
  S extends string,
  T extends string,
> = S extends `${infer L}${T}${infer R}` ? `${L}${R}` : never;
export type BeforeEventName = OmitSubstring<EventName & string, 'before-'>;
export type BeforeEventNameAndPrepend =
  CommonEventNameAndPrepend<BeforeEventName>;

export type ContextFunction<T, C extends Context = Context> = (ctx: C) => T;
export type OnContextFunction<C extends Context = Context> = ContextFunction<
  C,
  C
>;

export interface MappingStruct<
  T extends Record<string | number | symbol, any>,
  K extends keyof T,
> {
  type: K;
  data?: T[K];
}

export function GenerateMappingStruct<
  T extends Record<string | number | symbol, any>,
  K extends keyof T,
>(type: K, data?: T[K]): MappingStruct<T, K> {
  return {
    type,
    data,
  };
}

// Command stuff

export interface CommandRegisterConfig<D extends string = string> {
  def: D;
  desc?: string;
  config?: CommandConfigExtended;
  putOptions?: CommandPut.Config[];
}

export interface CommandConfigExtended extends Command.Config {
  empty?: boolean;
}

export interface CommandOptionConfig {
  name: string;
  desc: string;
  config?: Argv.OptionConfig;
}

export type CommandDefinitionFun = (cmd: Command) => Command;

export interface KoishiRouteDef {
  path: string;
  method:
    | 'get'
    | 'post'
    | 'put'
    | 'delete'
    | 'patch'
    | 'options'
    | 'head'
    | 'all';
}

export type KoaContext = ParameterizedContext<
  DefaultState,
  DefaultContext & RouterParamContext<DefaultState, DefaultContext>,
  any
>;
