import {
  BeforeEventName,
  CommandConfigExtended,
  CommandDefinitionFun,
  EventName,
  KoishiCommandDefinition,
  KoishiCommandPutDef,
  KoishiOnContextScope,
  OnContextFunction,
} from '../def';
import 'reflect-metadata';
import { Argv, Command, FieldCollector, Selection, Session } from 'koishi';
import { Metadata } from '../meta/metadata.decorators';
import { CommandPut, DoRegister } from '../registry';

// Register method

export const UseMiddleware = (prepend = false): MethodDecorator =>
  DoRegister.middleware(prepend);
export const UseEvent = (name: EventName, prepend = false): MethodDecorator =>
  DoRegister.event({ name, prepend });
export const UseBeforeEvent = (
  name: BeforeEventName,
  prepend = false,
): MethodDecorator => DoRegister.beforeEvent({ name, prepend });
export const UsePlugin = (): MethodDecorator => DoRegister.plugin();

export const Get = (path: string) => DoRegister.route({ path, method: 'get' });
export const Post = (path: string) =>
  DoRegister.route({ path, method: 'post' });
export const Put = (path: string) => DoRegister.route({ path, method: 'put' });
export const Delete = (path: string) =>
  DoRegister.route({ path, method: 'delete' });
export const Patch = (path: string) =>
  DoRegister.route({ path, method: 'patch' });
export const Options = (path: string) =>
  DoRegister.route({ path, method: 'options' });
export const Head = (path: string) =>
  DoRegister.route({ path, method: 'head' });
export const All = (path: string) => DoRegister.route({ path, method: 'all' });
export const Ws = DoRegister.ws;

export function UseCommand<D extends string>(
  def: D,
  config?: CommandConfigExtended,
): MethodDecorator;
export function UseCommand<D extends string>(
  def: D,
  desc: string,
  config?: CommandConfigExtended,
): MethodDecorator;
export function UseCommand(
  def: string,
  ...args: [CommandConfigExtended?] | [string, CommandConfigExtended?]
): MethodDecorator {
  const desc = typeof args[0] === 'string' ? (args.shift() as string) : '';
  const config = args[0] as CommandConfigExtended;
  return (obj, key: string, des) => {
    const putOptions: CommandPut.Config<keyof CommandPut.ConfigMap>[] =
      Reflect.getMetadata(KoishiCommandPutDef, obj.constructor, key) ||
      undefined;
    const metadataDec = DoRegister.command({
      def,
      desc,
      config,
      putOptions,
    });
    return metadataDec(obj, key, des);
  };
}

// Context scopes

export const OnContext = (
  ctxFun: OnContextFunction,
): MethodDecorator & ClassDecorator =>
  Metadata.append(KoishiOnContextScope, ctxFun);

export const OnAnywhere = () => OnContext((ctx) => ctx.any());

export const OnNowhere = () => OnContext((ctx) => ctx.never());

export const OnUser = (...values: string[]) =>
  OnContext((ctx) => ctx.user(...values));

export const OnSelf = (...values: string[]) =>
  OnContext((ctx) => ctx.self(...values));

export const OnGuild = (...values: string[]) =>
  OnContext((ctx) => ctx.guild(...values));

export const OnChannel = (...values: string[]) =>
  OnContext((ctx) => ctx.channel(...values));

export const OnPlatform = (...values: string[]) =>
  OnContext((ctx) => ctx.platform(...values));

export const OnPrivate = (...values: string[]) =>
  OnContext((ctx) => ctx.private(...values));

export const OnSelection = (selection: Selection) =>
  OnContext((ctx) => ctx.select(selection));

// Command definition

export const CommandDef = (
  def: CommandDefinitionFun,
): MethodDecorator & ClassDecorator =>
  Metadata.append(KoishiCommandDefinition, def);

export const CommandUse = <T extends Command, R extends any[]>(
  callback: (command: Command, ...args: R) => T,
  ...args: R
) => CommandDef((cmd) => callback(cmd, ...args));

export const CommandDescription = (desc: string) =>
  CommandDef((cmd) => {
    cmd.description = desc;
    return cmd;
  });

export const CommandAlias = (...names: string[]) =>
  CommandDef((cmd) => cmd.alias(...names));

export const CommandShortcut = (
  name: string | RegExp,
  config: Command.Shortcut = {},
) => CommandDef((cmd) => cmd.shortcut(name, config));

export const CommandUsage = (text: Command.Usage) =>
  CommandDef((cmd) => cmd.usage(text));

export const CommandExample = (text: string) =>
  CommandDef((cmd) => cmd.example(text));

export const CommandOption = (
  name: string,
  desc: string,
  config: Argv.OptionConfig = {},
) => CommandDef((cmd) => cmd.option(name, desc, config));

export const CommandUserFields = (fields: FieldCollector<'user'>) =>
  CommandDef((cmd) => cmd.userFields(fields));

export const CommandChannelFields = (fields: FieldCollector<'channel'>) =>
  CommandDef((cmd) => cmd.channelFields(fields));

export const CommandBefore = (callback: Command.Action, append = false) =>
  CommandDef((cmd) => cmd.before(callback, append));

export const CommandAction = (callback: Command.Action, prepend = false) =>
  CommandDef((cmd) => cmd.action(callback, prepend));

// Command put config

export const PutArgv = (field?: keyof Argv) =>
  field ? CommandPut.decorate('argvField', field) : CommandPut.decorate('argv');
export const PutSession = (field?: keyof Session) =>
  field ? CommandPut.decorate('sessionField', field) : PutArgv('session');
export const PutArg = (i: number) => CommandPut.decorate('arg', i);
export const PutArgs = () => CommandPut.decorate('args');
export const PutOption = (
  name: string,
  desc: string,
  config: Argv.OptionConfig = {},
) => CommandPut.decorate('option', { name, desc, config });

export const PutUser = (field: FieldCollector<'user'>) =>
  CommandPut.decorate('user', field);

export const PutChannel = (field: FieldCollector<'channel'>) =>
  CommandPut.decorate('channel', field);

export const PutGuild = (field: FieldCollector<'channel'>) =>
  CommandPut.decorate('guild', field);

export const PutUserName = (useDatabase = true) =>
  CommandPut.decorate('username', useDatabase);

export const PutUserId = () => PutSession('userId');
export const PutGuildId = () => PutSession('guildId');
export const PutGuildName = () => PutSession('guildName');
export const PutChannelId = () => PutSession('channelId');
export const PutChannelName = () => PutSession('channelName');
export const PutSelfId = () => PutSession('selfId');
export const PutBot = () => PutSession('bot');
export const PutNext = () => PutArgv('next');
