import yaml from 'yaml';
import fs from 'fs';
import { KoishiConfig, PackageConfig } from './def/interfaces';
import { Logger } from 'koishi';
import { exec } from 'child_process';
import { promisify } from 'util';
import loadJsonFile from 'load-json-file';
import path from 'path';
const execAsync = promisify(exec);

const logger = new Logger('bootstrap');

async function loadFromYaml(): Promise<KoishiConfig> {
  return yaml.parse(await fs.promises.readFile('./koishi.config.yml', 'utf-8'));
}

function loadFromJs(): KoishiConfig {
  return require('./koishi.config.js');
}

async function loadConfig(): Promise<KoishiConfig | undefined> {
  try {
    logger.info(`Reading config from ./koishi.config.yml.`);
    return await loadFromYaml();
  } catch (e) {
    logger.warn(
      `Failed reading from YAML: ${(e as any).toString()} , trying JS.`,
    );
    logger.info(`Reading config from ./koishi.config.js.`);
    try {
      return loadFromJs();
    } catch (e) {
      logger.warn(`Failed reading from JS: ${(e as any).toString()} .`);
      return;
    }
  }
}

async function getPackageJsonPackages() {
  const { dependencies } = await loadJsonFile<PackageConfig>('./package.json');
  return dependencies;
}

async function checkPluginExists(
  name: string,
  allowCommunity: boolean,
  allowOfficial: boolean,
) {
  const deps = await getPackageJsonPackages();
  const entry = Object.entries(deps).find(
    ([packageName, version]) =>
      packageName === name ||
      (allowOfficial && packageName === `@koishijs/plugin-${name}`) ||
      (allowCommunity && packageName.endsWith(`koishi-plugin-${name}`)),
  );
  if (!entry) {
    return;
  }
  try {
    const stat = await fs.promises.stat(path.join('node_modules', entry[0]));
    if (!stat?.isDirectory()) {
      return;
    }
    return entry[1];
  } catch (e) {
    return;
  }
}

async function npmInstall(name: string) {
  logger.info(`Installing package ${name}.`);
  try {
    await execAsync(`npm install --save-exact --loglevel error ${name}`);
    logger.info(`Package ${name} installed.`);
    return true;
  } catch (e) {
    logger.warn(`Package ${name} not found.`);
    return false;
  }
}
async function tryInstallPackages(names: string[]) {
  for (const name of names) {
    if (await npmInstall(name)) {
      return true;
    }
  }
  return false;
}

async function installPlugin(name: string, info: any) {
  const version: string = info.$version || info.$install;
  if (name.match(/^([\.\/~\\]|[A-Za-z]:[\/\\])/)) {
    logger.info(`Plugin ${name} is a local plugin, skipping.`);
    return;
  }
  logger.info(`Installing plugin ${name}@${version || 'unknown'}.`);
  const allowCommunity = !info.$official;
  const allowOfficial = !info.$community;
  if (!allowCommunity && !allowOfficial) {
    logger.warn(`Plugin ${name} is neither official nor community, skipping.`);
    return;
  }
  const existingPluginVersion = await checkPluginExists(
    name,
    allowCommunity,
    allowOfficial,
  );
  if (
    existingPluginVersion &&
    (existingPluginVersion === version || !version)
  ) {
    logger.info(`Plugin ${name}@${existingPluginVersion} exists, skipping.`);
    return;
  }
  const installList: string[] = [];
  const communityPrefix = 'koishi-plugin-';
  const officialPrefix = '@koishijs/plugin-';
  const installVersion = version || 'latest';
  if (name.includes(communityPrefix) || name.startsWith(officialPrefix)) {
    installList.push(`${name}@${installVersion}`);
  } else {
    if (allowOfficial) {
      installList.push(`${officialPrefix}${name}@${installVersion}`);
    }
    if (allowCommunity) {
      installList.push(`${communityPrefix}${name}@${installVersion}`);
    }
  }
  if (!installList.length) {
    logger.warn(`Plugin ${name} has nothing to install.`);
    return;
  }
  const result = await tryInstallPackages(installList);
  if (!result) {
    logger.error(`Plugin ${name}@${version || 'unknown'} install failed.`);
  }
}

async function main() {
  logger.info(`Bootstrapping`);
  const config = await loadConfig();
  const plugins = config?.plugins;
  if (!plugins) {
    logger.warn(`No plugins found, exiting.`);
    return;
  }
  logger.info(`Cleaning NPM cache.`);
  await execAsync(`npm cache clean --force`);
  for (const [name, info] of Object.entries(plugins)) {
    if (!info.$install) {
      continue;
    }
    const version =
      typeof info.$install !== 'boolean' ? info.$install : undefined;
    await installPlugin(name, info);
  }
  logger.info(`Bootstrap finished.`);
}
main();
