import yaml from 'yaml';
import fs from 'fs';
import { KoishiConfig, PackageConfig } from './def/interfaces';
import { Logger } from 'koishi';
import { exec } from 'child_process';
import { promisify } from 'util';
import loadJsonFile from 'load-json-file';
import path from 'path';
const execAsync = promisify(exec);

const logger = new Logger('bootstrap');

async function loadFromYaml(): Promise<KoishiConfig> {
  return yaml.parse(await fs.promises.readFile('./koishi.config.yml', 'utf-8'));
}

function loadFromJs(): KoishiConfig {
  return require('./koishi.config.js');
}

async function loadConfig(): Promise<KoishiConfig | undefined> {
  try {
    logger.info(`Reading config from ./koishi.config.yml.`);
    return await loadFromYaml();
  } catch (e) {
    logger.warn(
      `Failed reading from YAML: ${(e as any).toString()} , trying JS.`,
    );
    logger.info(`Reading config from ./koishi.config.js.`);
    try {
      return loadFromJs();
    } catch (e) {
      logger.warn(`Failed reading from JS: ${(e as any).toString()} .`);
      return;
    }
  }
}

async function getPackageJsonPackages() {
  const { dependencies } = await loadJsonFile<PackageConfig>('./package.json');
  return dependencies;
}

async function checkPluginExists(name: string) {
  const deps = await getPackageJsonPackages();
  const entry = Object.entries(deps).find(
    ([packageName, version]) =>
      packageName === `@koishijs/plugin-${name}` ||
      packageName.endsWith(`koishi-plugin-${name}`),
  );
  if (!entry) {
    return;
  }
  try {
    const stat = await fs.promises.stat(path.join('node_modules', entry[0]));
    if (!stat?.isDirectory()) {
      return;
    }
    return entry[1];
  } catch (e) {
    return;
  }
}

async function npmInstall(name: string) {
  logger.info(`Installing package ${name}.`);
  try {
    await execAsync(`npm install --loglevel error ${name}`);
    const installedVersionEntry = Object.entries(
      await getPackageJsonPackages(),
    ).find(([packageName]) => name.startsWith(packageName));
    logger.info(`Package ${name}@${installedVersionEntry![1]} installed.`);
    return true;
  } catch (e) {
    logger.warn(`Package ${name} not found.`);
    return false;
  }
}
async function tryInstallPackages(names: string[]) {
  for (const name of names) {
    if (await npmInstall(name)) {
      return true;
    }
  }
  return false;
}

async function installPlugin(name: string, version?: string) {
  logger.info(`Installing plugin ${name}@${version || 'unknown'}.`);
  const existingPluginVersion = await checkPluginExists(name);
  if (
    existingPluginVersion &&
    (existingPluginVersion === version ||
      !version ||
      version === 'latest' ||
      version === 'next')
  ) {
    logger.info(`Plugin ${name}@${existingPluginVersion} exists, skipping.`);
    return;
  }
  const installList: string[] = [];
  if (version) {
    installList.push(`@koishijs/plugin-${name}@${version}`);
    installList.push(`koishi-plugin-${name}@${version}`);
  } else {
    installList.push(`@koishijs/plugin-${name}@next`);
    installList.push(`@koishijs/plugin-${name}@latest`);
    installList.push(`koishi-plugin-${name}@next`);
    installList.push(`koishi-plugin-${name}@latest`);
  }
  const result = await tryInstallPackages(installList);
  if (!result) {
    logger.error(`Plugin ${name}@${version || 'unknown'} install failed.`);
  }
}

async function main() {
  logger.info(`Bootstrapping`);
  const config = await loadConfig();
  const plugins = config?.plugins;
  if (!plugins) {
    logger.warn(`No plugins found, exiting.`);
    return;
  }
  for (const [name, info] of Object.entries(plugins)) {
    if (!info.$install) {
      continue;
    }
    const version =
      typeof info.$install !== 'boolean' ? info.$install : undefined;
    await installPlugin(name, version);
  }
  logger.info(`Bootstrap finished.`);
}
main();
